<?php
/**
 * 2013 - 2023 Payplug SAS
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0).
 * It is available through the world-wide-web at this URL:
 * https://opensource.org/licenses/osl-3.0.php
 * If you are unable to obtain it through the world-wide-web, please send an email
 * to contact@payplug.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PayPlug module to newer
 * versions in the future.
 *
 * @author    Payplug SAS
 * @copyright 2013 - 2023 Payplug SAS
 * @license   https://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 *  International Registered Trademark & Property of Payplug SAS
 */

namespace PayPlug\src\repositories;

use PayPlug\src\application\dependencies\BaseClass;
use PayPlug\src\models\entities\CardEntity;

class CardRepository extends BaseClass
{
    protected $dependencies;

    private $cardEntity;
    private $configurationAdapter;
    private $constant;
    private $query;
    private $logger;
    private $toolsAdapter;
    private $validators;

    public function __construct(
        $configurationAdapter,
        $constant,
        $dependencies,
        $logger,
        $query,
        $tools
    ) {
        $this->cardEntity = new CardEntity();
        $this->configurationAdapter = $configurationAdapter;
        $this->constant = $constant;
        $this->dependencies = $dependencies;
        $this->logger = $logger;
        $this->query = $query;
        $this->toolsAdapter = $tools;
        $this->validators = $this->dependencies->getValidators();

        $isSandbox = $this->configurationAdapter->get(
            $this->dependencies->getConfigurationKey('sandboxMode')
        );
        $idCompany = $this->configurationAdapter->get(
            $this->dependencies->getConfigurationKey('companyId')
        );
        $this->logger->setProcess('card');

        $this->cardEntity
            ->setAllowedBrand(['mastercard', 'visa'])
            ->setFieldsRequired([])
            ->setFieldsSize([])
            ->setFieldsValidate([])
            ->setTable($this->dependencies->name . '_card')
            ->setIdentifier('');
        if ($idCompany && (!empty($idCompany))) {
            $this->cardEntity->setIdCompany((int) $idCompany);
        }

        if ($isSandbox) {
            $this->cardEntity->setIsSandbox((bool) $isSandbox);
        }
    }

    /**
     * @description confirm delete card message, called in front/cards.php
     *
     * @return mixed
     */
    public function confirmDeleteCardMessage()
    {
        return $this->dependencies->l('card.CardRepository.confirmDeleteCardMessage', 'cardrepository');
    }

    /**
     * @description Delete card
     *
     * @param int $id_customer
     * @param int $id_payplug_card
     *
     * @return bool
     */
    public function deleteCard($id_customer = 0, $id_payplug_card = 0)
    {
        if (!is_int($id_payplug_card) || !$id_payplug_card) {
            $this->logger->addLog('[deleteCard] Invalid argument given, $id_payplug_card must be a non null integer');

            return false;
        }

        if (!is_int($id_customer) || !$id_customer) {
            $this->logger->addLog('[deleteCard] Invalid argument given, $id_customer must be a non null integer');

            return false;
        }

        $card = $this->dependencies->getRepositories()['card']->get((int) $id_payplug_card);

        if (!$card) {
            $this->logger->addLog('[deleteCard] No Card found with payplug card id given: ' . $id_payplug_card);

            return false;
        }

        if ($card['id_customer'] != $id_customer) {
            $this->logger->addLog(
                'Error:  No Card found on [deleteCard] with payplug card id given: ' .
                '$id_payplug_card ' . (isset($id_payplug_card) ? $id_payplug_card : '')
            );

            return false;
        }

        $validate_expiration = $this->validators['card']->isValidExpiration((int) $card['exp_month'], (int) $card['exp_year']);
        if ($validate_expiration['result']) {
            if (!$this->deleteCardFromAPI($card['id_card'])) {
                return false;
            }
        }

        $delete = $this->dependencies->getRepositories()['card']->remove((int) $id_payplug_card);

        return (bool) $delete;
    }

    /**
     * @description Delete Card From API with a given ID Card
     *
     * @param string $id_card
     *
     * @return bool
     */
    public function deleteCardFromAPI($id_card = '')
    {
        if (!$id_card || !is_string($id_card)) {
            $this->logger->addLog('Error:  Bad parameters were passed to [deleteCardFromAPI]'
                . '$id_card: ' . json_encode($id_card));

            return false;
        }

        $delete = $this->dependencies->apiClass->deleteCard($id_card);

        if (200 == $delete['code']) {
            $json_answer = $delete['resource']['httpResponse'];

            if (isset($json_answer['object']) && $json_answer['object'] == 'error') {
                $message_log = 'Error occured while deleting the card' . $id_card . 'from the API [deleteCardFromAPI]';
                $this->logger->addLog($message_log, 'error');
                $this->logger->addLog('JSON answer: ' . json_encode($json_answer));

                return false;
            }
        }

        return true;
    }

    /**
     * @description Return successflull deleted card, called in front/cards.php
     *
     * @return mixed
     */
    public function deleteCardMessage()
    {
        return $this->dependencies->l('card.CardRepository.deleteCardMessage', 'cardrepository');
    }

    /**
     * @description Delete all cards for a given customer
     *
     * @param int $id_customer
     *
     * @return bool
     */
    public function deleteCards($id_customer)
    {
        if (!is_int($id_customer) || !isset($id_customer)) {
            $this->logger->addLog('Error: parameter $id_customer is empty or not integer [deleteCards]');

            return false;
        }

        $cardsToDelete = $this->getByCustomer((int) $id_customer, false);
        if (isset($cardsToDelete) && !empty($cardsToDelete) && sizeof($cardsToDelete)) {
            foreach ($cardsToDelete as $card) {
                if (!$this->deleteCard((int) $id_customer, (int) $card['id_payplug_card'])) {
                    $this->logger->addLog('Error : card can not be deleted [deleteCards]', 'error');
                    $this->logger->addLog('$card : ' . json_encode($card), 'debug');
                    $this->logger->addLog('$card[id_payplug_card] : ' . $card['id_payplug_card'], 'debug');
                    $this->logger->addLog('$id_customer: ' . $id_customer, 'debug');

                    return false;
                }
            }
        }

        return true;
    }

    /**
     * ## From classes/__PayPlugCard.php ##
     *
     * @description Get collection of cards for a given customer
     *
     * @param int  $id_customer
     * @param bool $active_only
     *
     * @return array
     */
    public function getByCustomer($id_customer, $active_only = false)
    {
        if (!is_int($id_customer) || !$id_customer) {
            $this->logger->addLog('[deleteCard] Invalid argument given, $id_payplug_card must be a non null integer');

            return [];
        }

        if (!is_bool($active_only)) {
            $this->logger->addLog('[deleteCard] Invalid argument given, $active_only must be a boolean');

            return [];
        }

        $id_company = $this->cardEntity->getIdCompany();
        $is_sandbox = $this->cardEntity->getIsSandbox();
        $cards = $this->dependencies
            ->getRepositories()['card']
            ->getAllByCustomer((int) $id_customer, (int) $id_company, (bool) $is_sandbox);

        if (!$cards) {
            $this->logger->addLog('$cards is null [getByCustomer]');

            return [];
        }

        // unset secret datas
        foreach ($cards as $key => &$card) {
            $is_expired = $this->validators['card']->isValidExpiration((int) $card['exp_month'], (int) $card['exp_year']);
            if (!$is_expired['result']) {
                $card['expired'] = true;
                if ($active_only) {
                    unset($cards[$key]);

                    continue;
                }
            } else {
                $card['expired'] = false;
            }
            $card['expiry_date'] = date(
                'm / y',
                mktime(0, 0, 0, (int) $card['exp_month'], 1, (int) $card['exp_year'])
            );

            unset($card['is_sandbox'], $card['id_card']);
        }

        return $cards;
    }

    /**
     * @description  Get Card detail form a Payment resource
     *
     * @param object $payment
     *
     * @return array
     */
    public function getCardDetailFromPayment($payment)
    {
        if (!is_object($payment) || !$payment) {
            $this->logger->addLog(
                '[getCardDetailFromPayment] Param $payment is invalid: ' . json_encode($payment),
                'error'
            );

            return [];
        }

        return [
            'last4' => (isset($payment->card->last4)
                && (!empty($payment->card->last4)))
                ? $payment->card->last4
                : null,
            'country' => (isset($payment->card->country)
                && (!empty($payment->card->country)))
                ? $payment->card->country
                : null,
            'exp_year' => (isset($payment->card->exp_year)
                && (!empty($payment->card->exp_year)))
                ? $payment->card->exp_year
                : null,
            'exp_month' => (isset($payment->card->exp_month)
                && (!empty($payment->card->exp_month)))
                ? $payment->card->exp_month
                : null,
            'brand' => (isset($payment->card->brand)
                && (!empty($payment->card->brand)))
                ? $payment->card->brand
                : null,
        ];
    }

    /**
     * @description Save a user card from a given payment resource
     *
     * @param $payment
     *
     * @return bool
     */
    public function saveCard($payment)
    {
        if (!isset($payment) || !is_object($payment)) {
            $this->logger->addLog('Parameter $payment is null or invalid [saveCard]', 'error');
            $this->logger->addLog('$payment: ' . json_encode($payment), 'debug');

            return false;
        }

        $customer_id = isset($payment->metadata['ID Client']) ? $payment->metadata['ID Client'] : $payment->metadata['Client'];

        $configurationClass = $this->dependencies->getPlugin()->getConfigurationClass();
        $company_id = (int) $configurationClass->getValue('company_id');

        $exists = $this->dependencies
            ->getRepositories()['card']
            ->exists((string) $payment->card->id, (int) $company_id, !(bool) $payment->is_live);
        if ($exists) {
            $this->logger->addLog('Error: this card with id_card = '
                . $payment->card->id . 'already exists [saveCard]');

            return false;
        }

        $register = $this->dependencies
            ->getRepositories()['card']
            ->set($payment->card, (int) $customer_id, (int) $company_id, !(bool) $payment->is_live);

        if (!$register) {
            $this->logger->addLog('Error : Unable to insert the card [saveCard].');

            return false;
        }

        return true;
    }
}
