<?php
/**
 * ISC License
 *
 * Copyright (c) 2025 idnovate.com
 * idnovate is a Registered Trademark & Property of idnovate.com, innovación y desarrollo SCP
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
 * INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * @author    idnovate
 * @copyright 2025 idnovate.com
 * @license   https://www.isc.org/licenses/ https://opensource.org/licenses/ISC ISC License
 */
if (!defined('_PS_VERSION_')) {
    exit;
}

class AdvancedPopup extends ObjectModel
{
    public $id_advancedpopup;
    public $id_shop;
    public $name;
    public $date_init;
    public $date_end;
    public $schedule;
    public $css_class;
    public $css;
    public $js;
    public $content;
    public $color_background;
    public $image_background;
    public $image;
    public $image_link;
    public $image_link_target;
    public $secs_to_display;
    public $secs_to_display_cart;
    public $secs_to_close;
    public $dont_display_again;
    public $priority = 0;
    public $back_opacity_value = 0.5;
    public $popup_height;
    public $popup_height_unit;
    public $popup_width;
    public $popup_width_unit;
    public $popup_padding;
    public $locked;
    public $responsive_min;
    public $responsive_max;
    public $show_customer_newsletter;
    public $show_customer_not_newsletter;
    public $show_on_view_page_nbr;
    public $show_every_nbr_hours;
    public $maximum_per_user;
    public $session;
    public $nb_products;
    public $nb_products_comparator;
    public $display_on_load;
    public $display_after_cart;
    public $display_on_exit;
    public $display_on_click;
    public $display_on_click_selector;
    public $display_inactive;
    public $secs_to_display_inactive;
    public $display_on_sticky;
    public $controller_exceptions = '';
    public $groups = '';
    public $zones = '';
    public $countries = '';
    public $categories = '';
    public $categories_selected = '';
    public $manufacturers = '';
    public $products = '';
    public $genders = '';
    public $customers = '';
    public $suppliers = '';
    public $cms = '';
    public $languages = '';
    public $attributes;
    public $features;
    public $close_on_background;
    public $blur_background;
    public $display_backdrop;
    public $open_effect;
    public $position = 5;
    public $cart_amount;
    public $cart_amount_from;
    public $cart_amount_to;
    public $display_url_string;
    public $display_referrer_string;
    public $display_ip_string;
    public $display_mobile = 1;
    public $display_tablet = 1;
    public $display_desktop = 1;
    public $product_specific_price;
    public $product_stock;
    public $product_stock_from;
    public $product_stock_to;
    public $product_price;
    public $product_price_from;
    public $product_price_to;
    public $product_oos;
    public $product_oos_value;
    public $sticky_display_close;
    public $sticky_position;
    public $sticky_rotation;
    public $sticky_content;
    public $active = 1;
    public $date_add;
    public $date_upd;

    protected $context;

    /**
     * @see ObjectModel::$definition
     */
    public static $definition = [
        'table' => 'advancedpopup',
        'primary' => 'id_advancedpopup',
        'multilang' => true,
        'fields' => [
            'id_shop' => ['type' => self::TYPE_INT, 'validate' => 'isUnsignedId', 'copy_post' => false],
            'name' => ['type' => self::TYPE_STRING, 'size' => 150, 'required' => true],
            'date_init' => ['type' => self::TYPE_DATE],
            'date_end' => ['type' => self::TYPE_DATE],
            'schedule' => ['type' => self::TYPE_STRING],
            'css_class' => ['type' => self::TYPE_STRING, 'lang' => true],
            'css' => ['type' => self::TYPE_HTML, 'size' => 65534, 'lang' => true],
            'js' => ['type' => self::TYPE_HTML, 'size' => 65534, 'lang' => true],
            'content' => ['type' => self::TYPE_HTML, 'size' => 65534, 'lang' => true],
            'color_background' => ['type' => self::TYPE_STRING],
            'image_background' => ['type' => self::TYPE_STRING, 'lang' => true],
            'image' => ['type' => self::TYPE_STRING, 'lang' => true],
            'image_link' => ['type' => self::TYPE_STRING, 'lang' => true],
            'image_link_target' => ['type' => self::TYPE_STRING],
            'secs_to_display' => ['type' => self::TYPE_INT],
            'secs_to_display_cart' => ['type' => self::TYPE_INT],
            'secs_to_close' => ['type' => self::TYPE_INT],
            'dont_display_again' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'priority' => ['type' => self::TYPE_INT, 'validate' => 'isInt', 'required' => true],
            'back_opacity_value' => ['type' => self::TYPE_FLOAT, 'validate' => 'isFloat'],
            'popup_height' => ['type' => self::TYPE_STRING, 'lang' => true],
            'popup_height_unit' => ['type' => self::TYPE_INT, 'validate' => 'isInt', 'lang' => true],
            'popup_width' => ['type' => self::TYPE_STRING, 'lang' => true],
            'popup_width_unit' => ['type' => self::TYPE_STRING, 'lang' => true],
            'popup_padding' => ['type' => self::TYPE_INT, 'validate' => 'isInt', 'lang' => true],
            'locked' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'responsive_min' => ['type' => self::TYPE_INT, 'validate' => 'isInt', 'lang' => true],
            'responsive_max' => ['type' => self::TYPE_INT, 'validate' => 'isInt', 'lang' => true],
            'show_customer_newsletter' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'show_customer_not_newsletter' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'show_on_view_page_nbr' => ['type' => self::TYPE_INT, 'validate' => 'isInt'],
            'show_every_nbr_hours' => ['type' => self::TYPE_STRING],
            'maximum_per_user' => ['type' => self::TYPE_INT, 'validate' => 'isInt'],
            'session' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'nb_products' => ['type' => self::TYPE_STRING],
            'nb_products_comparator' => ['type' => self::TYPE_INT, 'validate' => 'isInt'],
            'display_on_load' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'display_after_cart' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'display_on_exit' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'display_on_click' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'display_on_click_selector' => ['type' => self::TYPE_STRING, 'size' => 150],
            'display_inactive' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'secs_to_display_inactive' => ['type' => self::TYPE_INT],
            'display_on_sticky' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'controller_exceptions' => ['type' => self::TYPE_STRING, 'size' => 65534],
            'groups' => ['type' => self::TYPE_STRING],
            'countries' => ['type' => self::TYPE_STRING],
            'products' => ['type' => self::TYPE_STRING],
            'genders' => ['type' => self::TYPE_STRING],
            'customers' => ['type' => self::TYPE_STRING],
            'zones' => ['type' => self::TYPE_STRING],
            'categories' => ['type' => self::TYPE_STRING],
            'categories_selected' => ['type' => self::TYPE_STRING],
            'manufacturers' => ['type' => self::TYPE_STRING],
            'suppliers' => ['type' => self::TYPE_STRING],
            'cms' => ['type' => self::TYPE_STRING],
            'languages' => ['type' => self::TYPE_STRING],
            'attributes' => ['type' => self::TYPE_STRING],
            'features' => ['type' => self::TYPE_STRING],
            'close_on_background' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'blur_background' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'display_backdrop' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'open_effect' => ['type' => self::TYPE_STRING],
            'position' => ['type' => self::TYPE_INT, 'validate' => 'isInt'],
            'cart_amount' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'cart_amount_from' => ['type' => self::TYPE_FLOAT, 'validate' => 'isPrice'],
            'cart_amount_to' => ['type' => self::TYPE_FLOAT, 'validate' => 'isPrice'],
            'display_url_string' => ['type' => self::TYPE_STRING, 'lang' => true],
            'display_referrer_string' => ['type' => self::TYPE_STRING, 'lang' => true],
            'display_ip_string' => ['type' => self::TYPE_STRING],
            'display_mobile' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'display_tablet' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'display_desktop' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'product_specific_price' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'product_stock' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'product_stock_from' => ['type' => self::TYPE_NOTHING, 'validate' => 'isInt'],
            'product_stock_to' => ['type' => self::TYPE_NOTHING, 'validate' => 'isInt'],
            'product_price' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'product_price_from' => ['type' => self::TYPE_FLOAT, 'validate' => 'isPrice'],
            'product_price_to' => ['type' => self::TYPE_FLOAT, 'validate' => 'isPrice'],
            'product_oos' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'product_oos_value' => ['type' => self::TYPE_NOTHING, 'validate' => 'isInt'],
            'sticky_display_close' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'sticky_position' => ['type' => self::TYPE_NOTHING, 'validate' => 'isInt'],
            'sticky_rotation' => ['type' => self::TYPE_NOTHING, 'validate' => 'isInt'],
            'sticky_content' => ['type' => self::TYPE_HTML, 'size' => 65534, 'lang' => true],
            'active' => ['type' => self::TYPE_BOOL, 'validate' => 'isBool'],
            'date_add' => ['type' => self::TYPE_DATE, 'validate' => 'isDate', 'copy_post' => false],
            'date_upd' => ['type' => self::TYPE_DATE, 'validate' => 'isDate', 'copy_post' => false],
        ],
    ];

    public function __construct($id = null, $id_lang = null)
    {
        parent::__construct($id, $id_lang);

        $this->context = Context::getContext();

        if (!$id) {
            $this->date_init = date('Y-m-d H:i:s');
        }
    }

    public function add($autodate = true, $null_values = false)
    {
        $this->id_shop = ($this->id_shop) ?: Context::getContext()->shop->id;

        return parent::add($autodate, true);
    }

    public function save($null_values = false, $auto_date = true)
    {
        return parent::save(true, $auto_date);
    }

    public function update($null_values = false)
    {
        return parent::update(true);
    }

    public function toggleStatus()
    {
        parent::toggleStatus();

        return Db::getInstance()->execute('
            UPDATE `' . _DB_PREFIX_ . bqSQL($this->def['table']) . '`
            SET `date_upd` = NOW()
            WHERE `' . bqSQL($this->def['primary']) . '` = ' . (int) $this->id);
    }

    public function delete()
    {
        $languages = Language::getLanguages(false);
        foreach (['image', 'image_background'] as $type) {
            foreach ($languages as $language) {
                $image = $this->{$type}[(int) $language['id_lang']];
                if ($image) {
                    AdvancedPopupCreator::deleteImage(_PS_MODULE_DIR_ . AdvancedPopupCreator::$image_dir . $image);
                }
            }
        }

        return parent::delete();
    }

    public function getPopups($preview = false)
    {
        $query = 'SELECT *
            FROM `' . _DB_PREFIX_ . $this->def['table'] . '`
            INNER JOIN `' . _DB_PREFIX_ . $this->def['table'] . '_lang` ON `' . _DB_PREFIX_ . $this->def['table'] . '`.`id_advancedpopup` = `' . _DB_PREFIX_ . $this->def['table'] . '_lang`.`id_advancedpopup`
            WHERE `id_shop` = ' . (int) $this->context->shop->id . ' AND `id_lang` = ' . (int) $this->context->language->id;

        // Preview is the only one who asks for a specific popup, so no need to check filter neither mark it as seen
        if ($preview) {
            $token = Tools::getAdminToken('AdminAdvancedPopupCreatorPopups' . Tools::getValue('popupId'));
            if (Tools::getValue('token') !== $token) {
                return false;
            }

            $query .= ' AND ' . _DB_PREFIX_ . $this->def['table'] . '.`id_advancedpopup` = ' . Tools::getValue('popupId');

            return Db::getInstance()->executeS($query);
        }

        $laPopupsVisited = $this->getApcCookiePopups();

        // Reset popups with session enabled
        if (!isset($_COOKIE['apc_popup_session'])) {
            $resetQuery = 'SELECT `id_advancedpopup`
                FROM `' . _DB_PREFIX_ . $this->def['table'] . '`
                WHERE `id_shop` = ' . (int) $this->context->shop->id . ' AND `session` = 1';

            $popups = Db::getInstance()->executeS($resetQuery);

            foreach ($popups as $popup) {
                unset($laPopupsVisited[(int) $popup['id_advancedpopup']]);
            }

            setcookie('apc_popup_session', true, 0, '/');
            $this->setApcCookiePopups($laPopupsVisited);
        }

        if (isset($this->context->cart, $this->context->cart->id_address_delivery)
            && $this->context->cart->id_address_delivery != 0) {
            $zone = Address::getZoneById($this->context->cart->id_address_delivery);
        } else {
            $zone = Country::getIdZone($this->context->country->id);
        }

        $query .= ' AND `active` = 1
            AND (`zones` = ""
                OR `zones` IS NULL
                OR FIND_IN_SET("' . (int) $zone . '", `zones`))
            AND (`date_init` <= "' . date('Y-m-d H:i:s') . '" OR `date_init` = "' . date('Y-m-d H:i:s', 0) . '")
            AND (`date_end` >= "' . date('Y-m-d H:i:s') . '" OR `date_end` = "' . date('Y-m-d H:i:s', 0) . '")
            AND (`customers` = ""
                OR `customers` IS NULL
                OR FIND_IN_SET("' . (int) $this->context->customer->id . '", `customers`))
            AND (`languages` = ""
                OR `languages` IS NULL
                OR FIND_IN_SET("' . (int) $this->context->language->id . '", `languages`))
            AND (`display_ip_string` = ""
                OR `display_ip_string` IS NULL
                OR FIND_IN_SET("' . Tools::getRemoteAddr() . '", `display_ip_string`))
            AND (`display_url_string` = ""
                OR `display_url_string` IS NULL
                OR INSTR("' . pSQL($_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']) . '", `display_url_string`) > 0)
            ';

        if ((int) $this->context->customer->id_gender) {
            $query .= ' AND (`genders` = ""
                OR `genders` IS NULL
                OR FIND_IN_SET("' . (int) $this->context->customer->id_gender . '", `genders`))';
        } else {
            $query .= ' AND (`genders` = ""
                OR `genders` IS NULL)';
        }

        $customerGroups = Customer::getGroupsStatic((int) $this->context->customer->id);
        $query_groups = ' AND (`groups` = ""
            OR `groups` IS NULL ';
        foreach ($customerGroups as $customerGroup) {
            $query_groups .= ' OR FIND_IN_SET(' . $customerGroup . ', `groups`)';
        }
        $query_groups .= ')';
        $query .= $query_groups;

        if (version_compare(_PS_VERSION_, '1.6.0.11', '<')) {
            require_once _PS_TOOL_DIR_ . 'mobile_Detect/Mobile_Detect.php';
            $mobile = new Mobile_Detect();
        } else {
            $mobile = $this->context;
        }

        if ($mobile->isMobile()) {
            $query .= ' AND `display_mobile` = 1';
        }

        if ($mobile->isTablet()) {
            $query .= ' AND `display_tablet` = 1';
        }

        if (!$mobile->isTablet() && !$mobile->isMobile()) {
            $query .= ' AND `display_desktop` = 1';
        }

        $controller = AdvancedPopupCreator::getController();

        if ($controller === 'cms') {
            $query .= ' AND (`cms` = ""
                OR `cms` IS NULL
                OR FIND_IN_SET("' . (int) Tools::getValue('id_cms') . '", `cms`))';
        } else {
            $query .= ' AND (`cms` = ""
                OR `cms` IS NULL)';
        }

        $query .= ' AND (`controller_exceptions` = ""
            OR `controller_exceptions` IS NULL
            OR FIND_IN_SET("' . $controller . '", `controller_exceptions`))';

        if ((int) Tools::getValue('id_product')) {
            $query .= ' AND (`display_after_cart` = 1
                OR (`products` = ""
                OR `products` IS NULL
                OR FIND_IN_SET(' . (int) Tools::getValue('id_product') . ', `products`)))';
        } else {
            $query .= ' AND (`display_after_cart` = 1
                OR `products` = ""
                OR `products` IS NULL)';
        }

        if ($controller === 'product') {
            $laProductCategories = Product::getProductCategories((int) Tools::getValue('id_product'));
            $query_groups = ' AND (`categories` = ""
                OR `categories` IS NULL ';
            foreach ($laProductCategories as $laProductCategory) {
                $query_groups .= ' OR FIND_IN_SET(' . $laProductCategory . ', `categories`)';
            }
            $query_groups .= ')';
            $query .= $query_groups;
        } elseif ($controller === 'category') {
            $query .= ' AND (`display_after_cart` = 1
                OR (`categories` = ""
                OR `categories` IS NULL
                OR FIND_IN_SET("' . (int) Tools::getValue('id_category') . '", `categories`)))';
        } else {
            $query .= ' AND (`display_after_cart` = 1
                OR `categories` = ""
                OR `categories` IS NULL)';
        }

        if ($controller === 'product') {
            $product = new Product((int) Tools::getValue('id_product'));

            $query .= ' AND (`display_after_cart` = 1
                OR (`manufacturers` = ""
                OR `manufacturers` IS NULL
                OR FIND_IN_SET("' . (int) $product->id_manufacturer . '", `manufacturers`)))';

            $query .= ' AND (`display_after_cart` = 1
                OR (`suppliers` = ""
                OR `suppliers` IS NULL
                OR FIND_IN_SET("' . (int) $product->id_supplier . '", `suppliers`)))';
        } elseif ($controller === 'manufacturer') {
            $query .= ' AND (`display_after_cart` = 1
                OR (`manufacturers` = ""
                OR `manufacturers` IS NULL
                OR FIND_IN_SET("' . (int) Tools::getValue('id_manufacturer') . '", `manufacturers`)))';
        } elseif ($controller === 'supplier') {
            $query .= ' AND (`display_after_cart` = 1
                OR (`suppliers` = ""
                OR `suppliers` IS NULL
                OR FIND_IN_SET("' . (int) Tools::getValue('id_supplier') . '", `suppliers`)))';
        } else {
            $query .= ' AND (`display_after_cart` = 1
                OR `manufacturers` = ""
                OR `manufacturers` IS NULL)';
            $query .= ' AND (`display_after_cart` = 1
                OR `suppliers` = ""
                OR `suppliers` IS NULL)';
        }

        $query .= ' ORDER BY `priority` ASC;';

        $popups = Db::getInstance()->executeS($query);

        $validPopups = [];

        foreach ($popups as $popup) {
            /************* Pagenbr *************/
            $liShowOnPageView = (int) $popup['show_on_view_page_nbr'];
            $liPopupID = (int) $popup['id_advancedpopup'];

            // If 0 or 1, display the popup always
            if ((int) $liShowOnPageView > 0) {
                if (empty($laPopupsVisited)
                    || !isset($laPopupsVisited[$liPopupID])
                    || !isset($laPopupsVisited[$liPopupID]['visits'])) {
                    continue;
                }

                if ($laPopupsVisited[$liPopupID]['visits'] < $liShowOnPageView) {
                    continue;
                }
            }

            /************* Hoursnbr *************/
            // Run filter for without minutes defined
            if ((int) $popup['show_every_nbr_hours'] > 0) {
                $liNow = time();

                $liPopupID = (int) $popup['id_advancedpopup'];
                $liMinutesToShow = (int) $popup['show_every_nbr_hours'] * 60;

                if (!empty($laPopupsVisited)
                    && isset($laPopupsVisited[$liPopupID]['last_displayed'])) {
                    $liShowOnTime = $laPopupsVisited[$liPopupID]['last_displayed'] + $liMinutesToShow;

                    if ($liNow < $liShowOnTime) {
                        continue;
                    }
                }
            }

            // Times displayed
            if ((int) $popup['maximum_per_user'] > 0) {
                $liPopupID = (int) $popup['id_advancedpopup'];

                if (!empty($laPopupsVisited)
                    && isset($laPopupsVisited[$liPopupID]['times_displayed'])
                    && (int) $popup['maximum_per_user'] <= $laPopupsVisited[$liPopupID]['times_displayed']) {
                    continue;
                }
            }

            // Customer newsletter
            if ((int) $this->context->customer->id && ((int) $popup['show_customer_newsletter'] || (int) $popup['show_customer_not_newsletter'])) {
                $loCustomer = new Customer((int) $this->context->customer->id);

                if ((int) $popup['show_customer_newsletter'] && !(bool) $loCustomer->newsletter) {
                    continue;
                }

                if ((int) $popup['show_customer_not_newsletter'] && (bool) $loCustomer->newsletter) {
                    continue;
                }
            }

            // Don't show this message again
            if (isset($laPopupsVisited[$liPopupID]['last_displayed'])
                && $laPopupsVisited[$liPopupID]['last_displayed'] == (PHP_INT_MAX - 1)) {
                continue;
            }

            // Schedule
            if (!self::isShowableBySchedule($popup['schedule'])) {
                continue;
            }

            // Filter by attributes
            if ($popup['attributes']) {
                $attributesSelected = explode(',', $popup['attributes']);
                // if (array_filter($attributesSelected)) {
                if ($controller === 'product') {
                    // Get attributes group
                    $attributeGroups = [];
                    foreach ($attributesSelected as $attributeSelected) {
                        $attribute = new Attribute((int) $attributeSelected);
                        $attributeGroups[$attribute->id_attribute_group] = true;
                    }

                    $hasAttributeGroups = [];
                    $product = new Product((int) Tools::getValue('id_product'));
                    $productAttributeCombinations = $product->getAttributeCombinations();
                    foreach ($productAttributeCombinations as $productAttributeCombination) {
                        if (in_array((int) $productAttributeCombination['id_attribute'], $attributesSelected)) {
                            $hasAttributeGroups[(int) $productAttributeCombination['id_attribute_group']] = true;
                        }
                    }

                    if ($hasAttributeGroups !== $attributeGroups) {
                        continue;
                    }
                } else {
                    continue;
                }
                // }
            }

            // Filter by feature
            if ($popup['features']) {
                $featuresSelected = explode(',', $popup['features']);
                // if (array_filter($featuresSelected)) {
                if ($controller === 'product') {
                    // Get features group
                    $featureGroups = [];
                    foreach ($featuresSelected as $featureSelected) {
                        $feature = new FeatureValue((int) $featureSelected);
                        $featureGroups[$feature->id_feature] = true;
                    }

                    $hasFeatureGroups = [];
                    $productFeatures = Product::getFeaturesStatic((int) Tools::getValue('id_product'));
                    foreach ($productFeatures as $productFeature) {
                        if (in_array($productFeature['id_feature_value'], $featuresSelected)) {
                            $hasFeatureGroups[(int) $productFeature['id_feature']] = true;
                        }
                    }

                    if ($hasFeatureGroups !== $featureGroups) {
                        continue;
                    }
                } else {
                    continue;
                }
                // }
            }

            // Exclude products with special price
            if ($controller === 'product') {
                if ($popup['product_specific_price']) {
                    $specific_price = SpecificPrice::getSpecificPrice(Tools::getValue('id_product'), $this->context->shop->id, $this->context->currency->id, $this->context->country->id, $this->context->shop->id_shop_group, 1, null, $this->context->customer->id, (int) $this->context->cart->id);

                    if ($specific_price) {
                        continue;
                    }
                }
            }

            $validPopups[] = $popup;
        }

        return $validPopups;
    }

    public function getPopupToDisplay($availablePopups)
    {
        if (isset($this->context->cart, $this->context->cart->id_address_delivery) && $this->context->cart->id_address_delivery != 0) {
            $zone = Address::getZoneById($this->context->cart->id_address_delivery);
        } else {
            $zone = Country::getIdZone($this->context->country->id);
        }

        $availablePopups = (strpos($availablePopups, ',') !== false ? implode(',', array_map('intval', explode(',', $availablePopups))) : (int) $availablePopups);

        $query = 'SELECT *
            FROM `' . _DB_PREFIX_ . $this->def['table'] . '`
            INNER JOIN `' . _DB_PREFIX_ . $this->def['table'] . '_lang` ON `' . _DB_PREFIX_ . $this->def['table'] . '`.`id_advancedpopup` = `' . _DB_PREFIX_ . $this->def['table'] . '_lang`.`id_advancedpopup`
            WHERE `id_shop` = ' . (int) $this->context->shop->id . ' AND `id_lang` = ' . (int) $this->context->language->id
           . ' AND ' . _DB_PREFIX_ . $this->def['table'] . '.' . $this->def['primary'] . ' IN (' . $availablePopups . ')'
           . ' AND `active` = 1
            AND (`countries` = ""
                OR `countries` IS NULL
                OR FIND_IN_SET("' . (int) $this->context->country->id . '", `countries`))
            AND (`zones` = ""
                OR `zones` IS NULL
                OR FIND_IN_SET("' . (int) $zone . '", `zones`))
            AND (`responsive_min` = ""
                OR `responsive_min` IS NULL
                OR responsive_min > ' . (int) Tools::getValue('responsiveWidth') . ')
            AND (`responsive_max` = ""
                OR `responsive_max` IS NULL
                OR responsive_max < ' . (int) Tools::getValue('responsiveWidth') . ')            
           AND (`display_referrer_string` = ""
                OR `display_referrer_string` IS NULL
                OR INSTR("' . pSQL(urldecode(Tools::getValue('referrer'))) . '", `display_referrer_string`) > 0)';

        switch ((int) Tools::getValue('event')) {
            case 1:
                $query .= ' AND `display_on_load` = 1';
                break;

            case 2:
                $query .= ' AND `display_after_cart` = 1';
                break;

            case 3:
                $query .= ' AND `display_on_exit` = 1';
                break;

            case 4:
                $query .= ' AND `display_on_click` = 1';
                break;
        }

        $controller = AdvancedPopupCreator::getController();

        if ((int) Tools::getValue('event') !== 2) {
            if ((int) Tools::getValue('id_product')) {
                $query .= ' AND (`products` = ""
                    OR `products` IS NULL
                    OR FIND_IN_SET(' . (int) Tools::getValue('id_product') . ', `products`))';
            } else {
                $query .= ' AND (`products` = ""
                    OR `products` IS NULL)';
            }

            if ($controller === 'product') {
                $laProductCategories = Product::getProductCategories((int) Tools::getValue('id_product'));
                $query_groups = ' AND (`categories` = ""
                    OR `categories` IS NULL ';
                foreach ($laProductCategories as $laProductCategory) {
                    $query_groups .= ' OR FIND_IN_SET(' . $laProductCategory . ', `categories`)';
                }
                $query_groups .= ')';
                $query .= $query_groups;
            } elseif ($controller === 'category') {
                $query .= ' AND (`categories` = ""
                    OR `categories` IS NULL
                    OR FIND_IN_SET("' . (int) Tools::getValue('id_category') . '", `categories`))';
            } else {
                $query .= ' AND (`categories` = ""
                    OR `categories` IS NULL )';
            }

            if ($controller === 'product') {
                $product = new Product((int) Tools::getValue('id_product'));
            }

            if ($controller === 'product') {
                $query .= ' AND (`manufacturers` = ""
                    OR `manufacturers` IS NULL
                    OR FIND_IN_SET("' . $product->id_manufacturer . '", `manufacturers`))';
            } elseif ($controller === 'manufacturer') {
                $query .= ' AND (`manufacturers` = ""
                    OR `manufacturers` IS NULL
                    OR FIND_IN_SET("' . (int) Tools::getValue('id_manufacturer') . '", `manufacturers`))';
            } else {
                $query .= ' AND (`manufacturers` = ""
                    OR `manufacturers` IS NULL)';
            }

            if ($controller === 'product') {
                $query .= ' AND (`suppliers` = ""
                    OR `suppliers` IS NULL
                    OR FIND_IN_SET("' . $product->id_supplier . '", `suppliers`))';
            } elseif ($controller === 'supplier') {
                $query .= ' AND (`suppliers` = ""
                    OR `suppliers` IS NULL
                    OR FIND_IN_SET("' . (int) Tools::getValue('id_supplier') . '", `suppliers`))';
            } else {
                $query .= ' AND (`suppliers` = ""
                    OR `suppliers` IS NULL)';
            }
        } else {
            $product = new Product((int) Tools::getValue('id_product'));
            $query .= ' AND (`products` = ""
                OR `products` IS NULL
                OR FIND_IN_SET(' . (int) Tools::getValue('id_product') . ', `products`))';

            $laProductCategories = Product::getProductCategories((int) Tools::getValue('id_product'));
            $query .= ' AND (`categories` = ""
                OR `categories` IS NULL ';
            foreach ($laProductCategories as $laProductCategory) {
                $query .= ' OR FIND_IN_SET(' . $laProductCategory . ', `categories`)';
            }
            $query .= ')';

            $query .= ' AND (`manufacturers` = ""
                OR `manufacturers` IS NULL
                OR FIND_IN_SET("' . (int) $product->id_manufacturer . '", `manufacturers`))';
            $query .= ' AND (`suppliers` = ""
                OR `suppliers` IS NULL
                OR FIND_IN_SET("' . (int) $product->id_supplier . '", `suppliers`))';
        }

        $query .= ' ORDER BY `priority` ASC;';

        $popups = Db::getInstance()->executeS($query);

        $laPopupsVisited = $this->getApcCookiePopups();

        $validPopups = [];

        $cartAmount = $this->context->cart->getOrderTotal();
        $liNow = time();

        foreach ($popups as $popup) {
            /************* Hoursnbr *************/
            // Run filter for without minutes defined
            if ((int) $popup['show_every_nbr_hours'] > 0) {
                $liMinutesToShow = (int) $popup['show_every_nbr_hours'] * 60;

                if (!empty($laPopupsVisited)
                    && isset($laPopupsVisited[(int) $popup['id_advancedpopup']]['last_displayed'])) {
                    $liShowOnTime = $laPopupsVisited[(int) $popup['id_advancedpopup']]['last_displayed'] + $liMinutesToShow;

                    if ($liNow < $liShowOnTime) {
                        continue;
                    }
                }
            }

            // Times displayed
            if ((int) $popup['maximum_per_user'] > 0) {
                $liPopupID = (int) $popup['id_advancedpopup'];

                if (!empty($laPopupsVisited)
                    && isset($laPopupsVisited[$liPopupID]['times_displayed'])
                    && (int) $popup['maximum_per_user'] <= $laPopupsVisited[$liPopupID]['times_displayed']) {
                    continue;
                }
            }

            // Don't show this message again
            if (isset($laPopupsVisited[(int) $popup['id_advancedpopup']]['last_displayed'])
                && $laPopupsVisited[(int) $popup['id_advancedpopup']]['last_displayed'] == (PHP_INT_MAX - 1)) {
                continue;
            }

            /************* Cart amount *************/
            if ($popup['cart_amount']) {
                if ($popup['cart_amount_from']) {
                    $cartAmountFromConverted = AdvancedPopupCreator::convertPriceFull($popup['cart_amount_from'], null, $this->context->currency);
                    if ($cartAmount < $cartAmountFromConverted) {
                        continue;
                    }
                }

                if ($popup['cart_amount_to']) {
                    $cartAmountToConverted = AdvancedPopupCreator::convertPriceFull($popup['cart_amount_to'], null, $this->context->currency);
                    if ($cartAmount > $cartAmountToConverted) {
                        continue;
                    }
                }
            }

            /************* Number of products in the cart *************/
            if (isset($popup['nb_products'])) {
                $productsInCart = 0;
                foreach ($this->context->cart->getProducts() as $cartProduct) {
                    // Check category
                    $productCategories = Product::getProductCategories($cartProduct['id_product']);

                    if ($popup['categories_selected']) {
                        foreach ($productCategories as $productCategory) {
                            if (in_array($productCategory, explode(',', $popup['categories_selected']))) {
                                $productsInCart += $cartProduct['cart_quantity'];
                            }
                        }
                    } else {
                        $productsInCart += $cartProduct['cart_quantity'];
                    }
                }

                switch ($popup['nb_products_comparator']) {
                    case 1:
                        if ($productsInCart < $popup['nb_products']) {
                            continue 2;
                        }
                        break;

                    case 2:
                        if ($productsInCart != $popup['nb_products']) {
                            continue 2;
                        }
                        break;

                    case 3:
                        if ($productsInCart > $popup['nb_products']) {
                            continue 2;
                        }
                        break;
                }
            }

            if ($controller === 'product' || (int) Tools::getValue('event') === 2) {
                if ($popup['product_stock']) {
                    $stock = (int) Product::getQuantity((int) Tools::getValue('id_product'), (int) Tools::getValue('id_product_attribute'));
                    $productStockFrom = is_null($popup['product_stock_from']) ? PHP_INT_MIN : (int) $popup['product_stock_from'];
                    $productStockTo = is_null($popup['product_stock_to']) ? PHP_INT_MAX : (int) $popup['product_stock_to'];

                    if ($productStockFrom > $stock
                            || $productStockTo < $stock) {
                        continue;
                    }
                }

                if ($popup['product_price']) {
                    $productPrice = Product::getPriceStatic((int) Tools::getValue('id_product'), true, (int) Tools::getValue('id_product_attribute'), 6, null, false, true);
                    $productPrice = AdvancedPopupCreator::convertPriceFull($productPrice, null, $this->context->currency);
                    $productPriceFrom = is_null($popup['product_price_from']) ? PHP_INT_MIN : $popup['product_price_from'];
                    $productPriceTo = is_null($popup['product_price_to']) ? PHP_INT_MAX : $popup['product_price_to'];

                    if ($productPriceFrom > $productPrice
                            || $productPriceTo < $productPrice) {
                        continue;
                    }
                }

                if ($popup['product_oos']) {
                    $availableOutOfStock = (int) Product::isAvailableWhenOutOfStock(StockAvailable::outOfStock((int) Tools::getValue('id_product')));
                    if ((int) $popup['product_oos_value'] !== $availableOutOfStock) {
                        continue;
                    }
                }
            } else {
                if ($popup['product_stock']) {
                    continue;
                }

                if ($popup['product_price']) {
                    continue;
                }

                if ($popup['product_oos']) {
                    continue;
                }
            }

            if ((int) Tools::getValue('event') !== 1
                && (int) Tools::getValue('event') !== 4) {
                $validPopups[] = $popup;

                return $validPopups;
            }

            // Check if popup is displayed with different delay
            if ((int) Tools::getValue('event') === 1) {
                foreach ($validPopups as $validPopup) {
                    if ((int) $popup['secs_to_display'] == (int) $validPopup['secs_to_display']) {
                        continue;
                    }
                }
            }

            $validPopups[] = $popup;
        }

        return $validPopups;
    }

    public static function existExitPopups()
    {
        $query = 'SELECT count(*)
            FROM `' . _DB_PREFIX_ . 'advancedpopup`
            INNER JOIN `' . _DB_PREFIX_ . 'advancedpopup_lang` ON `' . _DB_PREFIX_ . 'advancedpopup`.`id_advancedpopup` = `' . _DB_PREFIX_ . 'advancedpopup_lang`.`id_advancedpopup`
            WHERE `id_shop` = ' . (int) Context::getContext()->shop->id . ' AND `id_lang` = ' . (int) Context::getContext()->language->id
            . ' AND `active` = 1
            AND `display_on_exit` = 1';

        return (bool) Db::getInstance()->getValue($query);
    }

    public static function isShowableBySchedule($schedule)
    {
        $schedule = json_decode($schedule);
        $dayOfWeek = date('w') - 1;
        if ($dayOfWeek < 0) {
            $dayOfWeek = 6;
        }

        if (is_array($schedule)) {
            if (is_object($schedule[$dayOfWeek]) && $schedule[$dayOfWeek]->isActive === true) {
                if ($schedule[$dayOfWeek]->timeFrom <= date('H:i')
                    && $schedule[$dayOfWeek]->timeTill > date('H:i')) {
                    return true;
                }

                return false;
            }

            return false;
        }

        return true;
    }

    /**
     * Update the visits from passed popups on each loading.
     *
     * array $laPopupsVisited The popups with visits
     * array The popups with visits +1 on all
     */
    public function updateVisits()
    {
        $query = 'SELECT *
            FROM `' . _DB_PREFIX_ . $this->def['table'] . '`
            INNER JOIN `' . _DB_PREFIX_ . $this->def['table'] . '_lang` ON `' . _DB_PREFIX_ . $this->def['table'] . '`.`id_advancedpopup` = `' . _DB_PREFIX_ . $this->def['table'] . '_lang`.`id_advancedpopup`
            WHERE `id_shop` = ' . (int) $this->context->shop->id . ' AND `id_lang` = ' . (int) $this->context->language->id;

        $query .= ' AND `active` = 1
            AND (`date_init` <= "' . date('Y-m-d H:i:s') . '" OR `date_init` = "' . date('Y-m-d H:i:s', 0) . '")
            AND (`date_end` >= "' . date('Y-m-d H:i:s') . '" OR `date_end` = "' . date('Y-m-d H:i:s', 0) . '")
            AND (`customers` = ""
                OR `customers` IS NULL
                OR FIND_IN_SET("' . (int) $this->context->customer->id . '", `customers`))
            AND (`languages` = ""
                OR `languages` IS NULL
                OR FIND_IN_SET("' . (int) $this->context->language->id . '", `languages`))
            AND (`display_ip_string` = ""
                OR `display_ip_string` IS NULL
                OR FIND_IN_SET("' . Tools::getRemoteAddr() . '", `display_ip_string`))
            ';

        if ((int) $this->context->customer->id_gender) {
            $query .= ' AND (`genders` = ""
                OR `genders` IS NULL
                OR FIND_IN_SET("' . (int) $this->context->customer->id_gender . '", `genders`))';
        } else {
            $query .= ' AND (`genders` = ""
                OR `genders` IS NULL)';
        }

        $customerGroups = Customer::getGroupsStatic((int) $this->context->customer->id);
        $query_groups = ' AND (`groups` = ""
            OR `groups` IS NULL ';
        foreach ($customerGroups as $customerGroup) {
            $query_groups .= ' OR FIND_IN_SET(' . $customerGroup . ', `groups`)';
        }
        $query_groups .= ')';
        $query .= $query_groups;

        if (version_compare(_PS_VERSION_, '1.6.0.11', '<')) {
            require_once _PS_TOOL_DIR_ . 'mobile_Detect/Mobile_Detect.php';
            $this->mobile_detect = new Mobile_Detect();

            if ($this->mobile_detect->isMobile()) {
                $query .= ' AND `display_mobile` = 1';
            }

            if ($this->mobile_detect->isTablet()) {
                $query .= ' AND `display_tablet` = 1';
            }

            if (!$this->mobile_detect->isTablet() && !$this->mobile_detect->isMobile()) {
                $query .= ' AND `display_desktop` = 1';
            }
        } else {
            if ($this->context->isMobile()) {
                $query .= ' AND `display_mobile` = 1';
            }

            if ($this->context->isTablet()) {
                $query .= ' AND `display_tablet` = 1';
            }

            if (!$this->context->isTablet() && !$this->context->isMobile()) {
                $query .= ' AND `display_desktop` = 1';
            }
        }

        $controller = AdvancedPopupCreator::getController();

        if ($controller === 'cms') {
            $query .= ' AND (`cms` = ""
                OR `cms` IS NULL
                OR FIND_IN_SET("' . (int) Tools::getValue('id_cms') . '", `cms`))';
        } else {
            $query .= ' AND (`cms` = ""
                OR `cms` IS NULL)';
        }

        $query .= ' AND (`controller_exceptions` = ""
            OR `controller_exceptions` IS NULL
            OR FIND_IN_SET("' . $controller . '", `controller_exceptions`))';

        if ((int) Tools::getValue('id_product')) {
            $query .= ' AND (`display_after_cart` = 1
                OR (`products` = ""
                    OR `products` IS NULL
                    OR FIND_IN_SET(' . (int) Tools::getValue('id_product') . ', `products`)))';
        } else {
            $query .= ' AND (`display_after_cart` = 1
                OR `products` = ""
                OR `products` IS NULL)';
        }

        if ($controller === 'product') {
            $laProductCategories = Product::getProductCategories((int) Tools::getValue('id_product'));
            $query_groups = ' AND (`categories` = ""
                OR `categories` IS NULL ';
            foreach ($laProductCategories as $laProductCategory) {
                $query_groups .= ' OR FIND_IN_SET(' . $laProductCategory . ', `categories`)';
            }
            $query_groups .= ')';
            $query .= $query_groups;
        } elseif ($controller === 'category') {
            $query .= ' AND (`display_after_cart` = 1
                OR (`categories` = ""
                    OR `categories` IS NULL
                    OR FIND_IN_SET("' . (int) Tools::getValue('id_category') . '", `categories`)))';
        } else {
            $query .= ' AND (`display_after_cart` = 1
                OR `categories` = ""
                OR `categories` IS NULL)';
        }

        if ($controller === 'product') {
            $product = new Product((int) Tools::getValue('id_product'));
        }

        if ($controller === 'product') {
            $query .= ' AND (`display_after_cart` = 1
                OR (`manufacturers` = ""
                    OR `manufacturers` IS NULL
                    OR FIND_IN_SET("' . $product->id_manufacturer . '", `manufacturers`)))';
        } elseif ($controller === 'manufacturer') {
            $query .= ' AND (`display_after_cart` = 1
                OR (`manufacturers` = ""
                    OR `manufacturers` IS NULL
                    OR FIND_IN_SET("' . (int) Tools::getValue('id_manufacturer') . '", `manufacturers`)))';
        } else {
            $query .= ' AND (`display_after_cart` = 1
                OR `manufacturers` = ""
                OR `manufacturers` IS NULL)';
        }

        if ($controller === 'product') {
            $query .= ' AND (`display_after_cart` = 1
                OR (`suppliers` = ""
                    OR `suppliers` IS NULL
                    OR FIND_IN_SET("' . $product->id_supplier . '", `suppliers`)))';
        } elseif ($controller === 'supplier') {
            $query .= ' AND (`display_after_cart` = 1
                OR (`suppliers` = ""
                    OR `suppliers` IS NULL
                    OR FIND_IN_SET("' . (int) Tools::getValue('id_supplier') . '", `suppliers`)))';
        } else {
            $query .= ' AND (`display_after_cart` = 1
                OR `suppliers` = ""
                OR `suppliers` IS NULL)';
        }

        $query .= ' ORDER BY `priority` ASC;';

        $popups = Db::getInstance()->executeS($query);

        if (!empty($popups)) {
            $laCookiePopupsVisited = $this->getApcCookiePopups();

            foreach ($popups as $popup) {
                if (isset($laCookiePopupsVisited[(int) $popup['id_advancedpopup']], $laCookiePopupsVisited[(int) $popup['id_advancedpopup']]['visits'])) {
                    ++$laCookiePopupsVisited[(int) $popup['id_advancedpopup']]['visits'];
                } else {
                    $laCookiePopupsVisited[(int) $popup['id_advancedpopup']]['visits'] = 1;
                }
            }

            $this->setApcCookiePopups($laCookiePopupsVisited);
        }
    }

    public function getApcCookiePopups()
    {
        if (Configuration::get('APC_COOKIE')) {
            return json_decode($_COOKIE['apc_popup'], true, 512, JSON_BIGINT_AS_STRING);
        }

        return json_decode($this->context->cookie->apc_popup, true, 512, JSON_BIGINT_AS_STRING);
    }

    public function setApcCookiePopups($value)
    {
        if (Configuration::get('APC_COOKIE')) {
            setcookie('apc_popup', json_encode($value), time() + 3600 * 24 * (int) Configuration::get('PS_COOKIE_LIFETIME_FO'), '/');
            $_COOKIE['apc_popup'] = json_encode($value);
        }

        return $this->context->cookie->apc_popup = json_encode($value);
    }

    /**
     * Delete images associated with the object
     */
    public function deletePopupImage($id_language)
    {
        if (!$this->id || !$id_language) {
            return false;
        }

        $lsDestination = _PS_MODULE_DIR_ . AdvancedPopupCreator::$image_dir . $this->image[(int) $id_language];

        if (file_exists($lsDestination) && !unlink($lsDestination)) {
            return false;
        }

        $this->image[(int) $id_language] = '';
        if (!$this->save()) {
            return false;
        }

        return true;
    }

    public static function getNbObjects()
    {
        $query = 'SELECT COUNT(a.`id_advancedpopup`) AS nb
                FROM `' . _DB_PREFIX_ . 'advancedpopup` a
                WHERE `id_shop` = ' . (int) Context::getContext()->shop->id;

        return (int) Db::getInstance(_PS_USE_SQL_SLAVE_)->getValue($query);
    }

    public static function generatePFG($id_pfg)
    {
        require_once _PS_MODULE_DIR_ . 'powerfulformgenerator/classes/PFGRenderer.php';

        $renderer = new PFGRenderer($id_pfg);
        if ($renderer->isAllowed(true)) {
            return $renderer->displayForm();
        }

        return false;
    }
}
