<?php
/**
 * ISC License
 *
 * Copyright (c) 2025 idnovate.com
 * idnovate is a Registered Trademark & Property of idnovate.com, innovación y desarrollo SCP
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
 * INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * @author    idnovate
 * @copyright 2025 idnovate.com
 * @license   https://www.isc.org/licenses/ https://opensource.org/licenses/ISC ISC License
 */
if (!defined('_PS_VERSION_')) {
    exit;
}

class AdminAdvancedPopupCreatorPopupsController extends ModuleAdminController
{
    protected $_defaultOrderBy = 'date_add';
    protected $_defaultOrderWay = 'DESC';
    protected $orderBy = 'id_advancedpopup';
    protected $orderWay = 'ASC';
    protected $isShopSelected = true;

    public function __construct()
    {
        $this->bootstrap = true;
        $this->table = 'advancedpopup';
        $this->className = 'AdvancedPopup';
        $this->tabClassName = 'AdminAdvancedPopupCreatorPopups';
        $this->lang = true;
        $this->addRowAction('edit');
        $this->addRowAction('delete');
        $this->_orderWay = $this->_defaultOrderWay;

        parent::__construct();

        $this->bulk_actions = [
            'delete' => [
                'text' => $this->module->l('Delete selected'),
                'confirm' => $this->module->l('Delete selected items?'),
                'icon' => 'icon-trash',
            ],
        ];

        $this->context = Context::getContext();

        $this->default_form_language = $this->context->language->id;

        $this->fields_list = [
            'id_advancedpopup' => [
                'title' => $this->module->l('ID'),
                'align' => 'text-center',
                'class' => 'fixed-width-xs',
            ],
            'active' => [
                'title' => $this->module->l('Active'),
                'align' => 'text-center',
                'active' => 'status',
                'type' => 'bool',
                'callback' => 'printActiveIcon',
            ],
            'name' => [
                'title' => $this->module->l('Name'),
                'filter_key' => 'a!name',
            ],
            'date_init' => [
                'title' => $this->module->l('From'),
                'align' => 'text-center',
                'type' => 'datetime',
                'callback' => 'printValidDate',
            ],
            'date_end' => [
                'title' => $this->module->l('To'),
                'align' => 'text-center',
                'type' => 'datetime',
                'callback' => 'printValidDate',
            ],
            'date_add' => [
                'title' => $this->module->l('Valid'),
                'align' => 'text-center',
                'search' => false,
                'callback' => 'printValidIcon',
            ],
            /*'priority' => [
                'title' => $this->module->l('Priority'),
                'align' => 'text-center',
            ],*/
            'display_on_load' => [
                'title' => $this->module->l('Page load'),
                'align' => 'text-center',
                'callback' => 'printCheckIcon',
                'orderby' => false,
                'search' => false,
            ],
            'display_after_cart' => [
                'title' => $this->module->l('Product added'),
                'align' => 'text-center',
                'callback' => 'printCheckIcon',
                'orderby' => false,
                'search' => false,
            ],
            'display_on_exit' => [
                'title' => $this->module->l('On exit'),
                'align' => 'text-center',
                'callback' => 'printCheckIcon',
                'orderby' => false,
                'search' => false,
            ],
            'display_inactive' => [
                'title' => $this->module->l('Inactivity'),
                'align' => 'text-center',
                'callback' => 'printCheckIcon',
                'orderby' => false,
                'search' => false,
            ],
            'display_on_click' => [
                'title' => $this->module->l('Click on element'),
                'align' => 'text-center',
                'callback' => 'printCheckIcon',
                'orderby' => false,
                'search' => false,
            ],
            'display_on_sticky' => [
                'title' => $this->module->l('Click on sticky'),
                'align' => 'text-center',
                'callback' => 'printCheckIcon',
                'orderby' => false,
                'search' => false,
            ],
            'show_every_nbr_hours' => [
                'title' => $this->module->l('Display every X minutes'),
                'align' => 'text-center',
                'orderby' => false,
                'search' => false,
            ],
        ];

        $this->_select .= '1 as on_load, 1 as after_cart, 1 as on_exit, 1 as inactive, 1 as on_click';

        if (Shop::isFeatureActive() && (Shop::getContext() == Shop::CONTEXT_ALL || Shop::getContext() == Shop::CONTEXT_GROUP)) {
            $this->isShopSelected = false;
        }

        if (!Shop::isFeatureActive()) {
            $this->shopLinkType = '';
        } else {
            $this->shopLinkType = 'shop';
        }
    }

    public function setMedia($isNewTheme = false)
    {
        parent::setMedia($isNewTheme);

        if ($this->display) {
            $this->context->controller->addCSS(_MODULE_DIR_ . $this->module->name . '/lib/CodeMirror/lib/codemirror.css');
            $this->context->controller->addCSS(_MODULE_DIR_ . $this->module->name . '/lib/CodeMirror/theme/monokai.css');

            $this->addJqueryPlugin(['typewatch', 'autocomplete']);
            $this->addJqueryUI(['ui.datepicker', 'ui.button', 'ui.sortable', 'ui.droppable']);

            if (version_compare(_PS_VERSION_, '1.6', '>=')) {
                $this->context->controller->addJS(_MODULE_DIR_ . $this->module->name . '/views/js/advancedpopup-admin.js', false);
            } else {
                // $this->context->controller->addJS(_MODULE_DIR_.$this->module->name.'/views/js/bootstrap-tab.js');
                $this->context->controller->addJS(_MODULE_DIR_ . $this->module->name . '/views/js/advancedpopup-admin15.js', false);
            }

            $this->context->controller->addJS(_MODULE_DIR_ . $this->module->name . '/views/js/tabs.js', false);
            $this->context->controller->addJS(_MODULE_DIR_ . $this->module->name . '/lib/CodeMirror/lib/codemirror.js', false);
            $this->context->controller->addJS(_MODULE_DIR_ . $this->module->name . '/lib/CodeMirror/addon/display/autorefresh.js', false);
            $this->context->controller->addJS(_MODULE_DIR_ . $this->module->name . '/lib/CodeMirror/mode/css/css.js', false);
            $this->context->controller->addJS(_MODULE_DIR_ . $this->module->name . '/lib/CodeMirror/mode/javascript/javascript.js');
        }

        if (version_compare(_PS_VERSION_, '1.6', '>=')) {
            $this->context->controller->addCSS(_MODULE_DIR_ . $this->module->name . '/views/css/advancedpopup-admin.css');
        } else {
            $this->context->controller->addCSS(_MODULE_DIR_ . $this->module->name . '/views/css/advancedpopup-admin15.css');
            $this->context->controller->addCSS(_MODULE_DIR_ . $this->module->name . '/views/css/tabs.css');
        }
    }

    public function initContent()
    {
        if ($this->action == 'select_delete') {
            $this->context->smarty->assign([
                'delete_form' => true,
                'url_delete' => htmlentities($_SERVER['REQUEST_URI']),
                'boxes' => $this->boxes,
            ]);
        }

        if (version_compare($this->module->version, $this->module->getDatabaseVersion(), '>')) {
            return $this->errors[] = $this->module->l('Upgrade available');
        }

        parent::initContent();

        if (!$this->display) {
            $html = '';
            if (version_compare(_PS_VERSION_, '1.6', '>=')) {
                if (isset($this->module)) {
                    $module = $this->module;
                } else {
                    $module = $this;
                }

                $default_iso_code = 'en';
                $local_path = $module->getLocalPath();

                if ($this->context->language->iso_code == 'ag'
                    || $this->context->language->iso_code == 'cb'
                    || $this->context->language->iso_code == 'mx'
                    || $this->context->language->iso_code == 'pe'
                    || $this->context->language->iso_code == 've') {
                    $iso_code = 'es';
                } elseif ($this->context->language->iso_code == 'qc') {
                    $iso_code = 'fr';
                } elseif ($this->context->language->iso_code == 'br') {
                    $iso_code = 'pt';
                } elseif ($this->context->language->iso_code == 'gb') {
                    $iso_code = 'en';
                } else {
                    $iso_code = $this->context->language->iso_code;
                }

                $readme = null;
                if (file_exists($local_path . '/readme_' . $iso_code . '.pdf')) {
                    $readme = 'readme_' . $iso_code . '.pdf';
                } elseif (file_exists($local_path . '/readme_' . $default_iso_code . '.pdf')) {
                    $readme = 'readme_' . $default_iso_code . '.pdf';
                }

                $this->context->smarty->assign([
                    'support_id' => $module->addons_id_product,
                    'readme' => $readme,
                    'this_path' => $module->getPathUri(),
                ]);

                if (file_exists($local_path . '/views/templates/admin/company/information_' . $iso_code . '.tpl')) {
                    $html .= $this->context->smarty->fetch($local_path . '/views/templates/admin/company/information_' . $iso_code . '.tpl');
                } elseif (file_exists($local_path . '/views/templates/admin/company/information_' . $default_iso_code . '.tpl')) {
                    $html .= $this->context->smarty->fetch($local_path . '/views/templates/admin/company/information_' . $default_iso_code . '.tpl');
                }
            }

            $this->content .= $html;

            $this->context->smarty->assign([
                'content' => $this->content,
            ]);
        }
    }

    public function initToolbarTitle()
    {
        parent::initToolbarTitle();

        switch ($this->display) {
            case '':
            case 'list':
                array_pop($this->toolbar_title);
                $this->toolbar_title[] = $this->module->l('Manage Advanced Popup Creator Configuration');
                break;
            case 'view':
                if (($object = $this->loadObject(true)) && Validate::isLoadedObject($object)) {
                    array_pop($this->toolbar_title);
                    $this->toolbar_title[] = sprintf($this->module->l('Configuration: %s'), $object->name);
                }
                break;
            case 'add':
            case 'edit':
                array_pop($this->toolbar_title);
                if (($object = $this->loadObject(true)) && Validate::isLoadedObject($object)) {
                    $this->toolbar_title[] = sprintf($this->module->l('Editing popup: %s'), $object->name);
                } else {
                    $this->toolbar_title[] = $this->module->l('New popup');
                }
                break;
        }
    }

    public function initPageHeaderToolbar()
    {
        parent::initPageHeaderToolbar();

        if (empty($this->display)) {
            $this->page_header_toolbar_btn['desc-module-new'] = [
                'href' => 'index.php?controller=' . $this->tabClassName . '&add' . $this->table . '&token=' . Tools::getAdminTokenLite($this->tabClassName),
                'desc' => $this->module->l('New'),
                'icon' => 'process-icon-new',
            ];

            $this->page_header_toolbar_btn['desc-module-translate'] = [
                'href' => '#',
                'desc' => $this->module->l('Translate'),
                'modal_target' => '#moduleTradLangSelect',
                'icon' => 'process-icon-flag',
            ];

            $this->page_header_toolbar_btn['desc-module-hook'] = [
                'href' => 'index.php?tab=AdminModulesPositions&token=' . Tools::getAdminTokenLite('AdminModulesPositions') . '&show_modules=' . Module::getModuleIdByName($this->module->name),
                'desc' => $this->module->l('Manage hooks'),
                'icon' => 'process-icon-anchor',
            ];
        }

        $this->context->smarty->clearAssign('help_link', '');
    }

    public function initModal()
    {
        parent::initModal();

        $languages = Language::getLanguages(false);
        $translateLinks = [];

        if (version_compare(_PS_VERSION_, '1.7.2.1', '>=')) {
            $module = Module::getInstanceByName($this->module->name);
            $isNewTranslateSystem = $module->isUsingNewTranslationSystem();
            $link = Context::getContext()->link;
            foreach ($languages as $lang) {
                if ($isNewTranslateSystem) {
                    $translateLinks[$lang['iso_code']] = $link->getAdminLink('AdminTranslationSf', true, [
                        'lang' => $lang['iso_code'],
                        'type' => 'modules',
                        'selected' => $module->name,
                        'locale' => $lang['locale'],
                    ]);
                } else {
                    $translateLinks[$lang['iso_code']] = $link->getAdminLink('AdminTranslations', true, [], [
                        'type' => 'modules',
                        'module' => $module->name,
                        'lang' => $lang['iso_code'],
                    ]);
                }
            }
        }

        $this->context->smarty->assign([
            'trad_link' => 'index.php?tab=AdminTranslations&token=' . Tools::getAdminTokenLite('AdminTranslations') . '&type=modules&module=' . $this->module->name . '&lang=',
            'module_languages' => $languages,
            'module_name' => $this->module->name,
            'translateLinks' => $translateLinks,
        ]);

        $modal_content = $this->context->smarty->fetch($this->module->getLocalPath() . 'views/templates/admin/modal_translation.tpl');

        $this->modals[] = [
            'modal_id' => 'moduleTradLangSelect',
            'modal_class' => 'modal-sm',
            'modal_title' => $this->module->l('Translate this module'),
            'modal_content' => $modal_content,
        ];
    }

    public function initProcess()
    {
        parent::initProcess();

        if (version_compare(_PS_VERSION_, '1.7', '<')) {
            if (Tools::getIsset('duplicate' . $this->table)) {
                if ((int) $this->tabAccess['add'] === 1) {
                    $this->action = 'duplicate';
                } else {
                    $this->errors[] = Tools::displayError('You do not have permission to add this.');
                }
            }
        } else {
            if (Tools::getIsset('duplicate' . $this->table)) {
                if ($this->access('add')) {
                    $this->action = 'duplicate';
                } else {
                    $this->errors[] = Tools::displayError('You do not have permission to add this.');
                }
            }
        }
    }

    public function postProcess()
    {
        if (Tools::isSubmit('submitAdd' . $this->table)) {
            if (($object = $this->loadObject(true)) || Validate::isLoadedObject($object)) {
                if (!(int) Tools::getValue('display_on_load')
                    && !(int) Tools::getValue('display_on_exit')
                    && !(int) Tools::getValue('display_after_cart')
                    && !(int) Tools::getValue('display_on_click')
                    && !(int) Tools::getValue('display_inactive')
                    && !(int) Tools::getValue('display_on_sticky')) {
                    $this->errors[] = $this->module->l('You must define when do you want to display the popup');
                }

                if ((int) Tools::getValue('display_on_click')
                    && !Tools::getValue('display_on_click_selector')) {
                    $this->errors[] = $this->module->l('You must define which selector fires the popup');
                }

                // These fields are int or empty, but we can't define them as INT because it sets 0 by default
                $stringFields = ['nb_products', 'show_every_nbr_hours', 'secs_to_display', 'secs_to_display_cart', 'secs_to_close'];

                foreach ($stringFields as $stringField) {
                    if (!Validate::isInt(Tools::getValue($stringField)) && Tools::getValue($stringField)) {
                        $this->errors[] = sprintf('The %s field is invalid', $object->displayFieldName($stringField, get_class($object)));
                    }
                }

                // Content
                $content = false;
                foreach (Language::getLanguages(false) as $language) {
                    if (strpos(Tools::getValue('content_' . $language['id_lang']), 'mojo/signup-forms/Loader') !== false
                        || (strpos(Tools::getValue('content_' . $language['id_lang']), '<iframe') !== false
                            // && !Configuration::get('APC_IFRAMES')
                            && !Configuration::get('PS_ALLOW_HTML_IFRAME'))) {
                        $this->errors[] = $this->module->l('To use "iframe", enable the feature in Shop Parameters > General');
                    }

                    if (Tools::getValue('content_' . $language['id_lang'])) {
                        $content = true;
                        break;
                    }

                    if (Tools::fileAttachment('image_' . $language['id_lang'])
                        || Tools::fileAttachment('image_background_' . $language['id_lang'])) {
                        $content = true;
                        break;
                    }

                    if (($object->image && $object->image[$language['id_lang']])
                        || ($object->image_background && $object->image_background[$language['id_lang']])) {
                        $content = true;
                        break;
                    }
                }

                if (!$content) {
                    $this->errors[] = $this->module->l('You must set some content at least in one language');
                }

                // Date from
                if (!Tools::getValue('date_init')) {
                    $_POST['date_init'] = date('Y-m-d H:i:s');
                }

                // Secs
                if ((int) Tools::getValue('secs_to_close') < 0) {
                    $this->errors[] = $this->module->l("Field 'Secs to close' can not be less than 0.");
                }

                // Priority
                if ((int) Tools::getValue('priority') < 0) {
                    $this->errors[] = $this->module->l("Field 'Priority' can not be less than 0.");
                }

                // Opacity
                if ((float) Tools::getValue('back_opacity_value') < 0 || (float) Tools::getValue('back_opacity_value') > 1) {
                    $this->errors[] = $this->module->l("Field 'Opacity' must be a number between 0 and 1.");
                }

                // Nbr of pages
                if ((int) Tools::getValue('show_on_view_page_nbr') < 0) {
                    $this->errors[] = $this->module->l("Field 'Nbr of pages before show' can not be less than 1.");
                }

                foreach (Language::getLanguages(false) as $language) {
                    // Height
                    if ((int) Tools::getValue('popup_height_' . $language['id_lang']) < 0) {
                        $this->errors[] = $this->module->l("Field 'Popup height' can not be less than 0.");
                    }

                    if ((int) Tools::getValue('popup_height_' . $language['id_lang']) > 0
                        && !Tools::getValue('popup_height_unit_' . $language['id_lang'])) {
                        $this->errors[] = $this->module->l("You need to define a unit for 'Popup height'.");
                    }

                    if ((int) Tools::getValue('popup_height_' . $language['id_lang']) > 100
                        && (int) Tools::getValue('popup_height_unit_' . $language['id_lang']) === 2) {
                        $this->errors[] = $this->module->l("'Popup height' percentage can not be higher than 100.");
                    }

                    // Width
                    if ((int) Tools::getValue('popup_width_' . $language['id_lang']) < 0) {
                        $this->errors[] = $this->module->l("Field 'Popup width' can not be less than 0.");
                    }

                    if ((int) Tools::getValue('popup_width_' . $language['id_lang']) > 0
                        && !Tools::getValue('popup_width_unit_' . $language['id_lang'])) {
                        $this->errors[] = $this->module->l("You need to define a unit for 'Popup width'.");
                    }

                    if ((int) Tools::getValue('popup_width_' . $language['id_lang']) > 100
                        && (int) Tools::getValue('popup_width_unit_' . $language['id_lang']) === 2) {
                        $this->errors[] = $this->module->l("'Popup width' percentage can not be higher than 100.");
                    }

                    // Padding
                    if ((int) Tools::getValue('popup_padding_' . $language['id_lang']) < 0) {
                        $this->errors[] = $this->module->l("Field 'Popup padding' can not be less than 0.");
                        break;
                    }
                }

                // Minutes
                if ((int) Tools::getValue('show_every_nbr_hours') < 0) {
                    $this->errors[] = $this->module->l("Field 'Show every minutes' can not be less than 0.");
                }

                // Responsive
                foreach (Language::getLanguages(false) as $language) {
                    if ((int) Tools::getValue('responsive_min_' . $language['id_lang']) != 0
                        && (int) Tools::getValue('responsive_max_' . $language['id_lang']) != 0
                        && (int) Tools::getValue('responsive_min_' . $language['id_lang']) > (int) Tools::getValue('responsive_max_' . $language['id_lang'])) {
                        $this->errors[] = $this->module->l('Minimum screen size can not be higher than maximum');
                        break;
                    }
                }

                // Newsletter
                if ((int) Tools::getValue('show_customer_newsletter')
                    && (int) Tools::getValue('show_customer_not_newsletter')) {
                    $this->errors[] = $this->module->l("You can not enable 'Display popup only to customers subscribed to newsletter' and 'Display popup only to customers NOT subscribed to newsletter' at the same time");
                }

                // Jquery selector
                if (Tools::getValue('display_on_click_selector') && Tools::substr(Tools::getValue('display_on_click_selector'), 0, 2) != '$(') {
                    $this->errors[] = $this->module->l('Incorrect jQuery selector format');
                }

                // Cart amount
                if ((float) Tools::getValue('cart_amount_from') > (float) Tools::getValue('cart_amount_to')) {
                    $this->errors[] = $this->module->l('"Cart amount from" can not be higher than "Cart amount to"');
                }

                // String in URL
                if ((int) Tools::getValue('display_url_string')) {
                    $value = false;
                    foreach (Language::getLanguages(false) as $language) {
                        if (Tools::getValue('display_url_string_' . $language['id_lang'])) {
                            $value = true;
                        }
                    }

                    if (!$value) {
                        $this->errors[] = $this->module->l('You need to define a string in URL');
                    }
                }

                if ((int) Tools::getValue('display_referrer_string')) {
                    $value = false;
                    foreach (Language::getLanguages(false) as $language) {
                        if (Tools::getValue('display_referrer_string_' . $language['id_lang'])) {
                            $value = true;
                        }
                    }

                    if (!$value) {
                        $this->errors[] = $this->module->l('You need to define a referrer');
                    }
                }

                // IPs
                if ((int) Tools::getValue('display_ip_string')) {
                    if (!Tools::getValue('display_ip_string')) {
                        $this->errors[] = $this->module->l('You need to define an IP');
                    }

                    foreach (explode(';', Tools::getValue('display_ip_string')) as $ip) {
                        if ($ip && !filter_var($ip, FILTER_VALIDATE_IP)) {
                            $this->errors[] = $ip . ' ' . $this->module->l('is not a valid IP');
                        }
                    }
                }

                // Product stock
                if ((int) Tools::getValue('product_stock_from')
                    && (int) Tools::getValue('product_stock_to')
                    && (int) Tools::getValue('product_stock_from') > (int) Tools::getValue('product_stock_to')) {
                    $this->errors[] = $this->module->l('"Product stock from" can not be higher than "Product stock to"');
                }

                // Product price
                if ((int) Tools::getValue('product_price_from')
                    && (float) Tools::getValue('product_price_to')
                    && (float) Tools::getValue('product_price_from') > (float) Tools::getValue('product_price_to')) {
                    $this->errors[] = $this->module->l('"Product price from" can not be higher than "Product price to"');
                }
            }

            $switchFields = [
                'switch_controller_exceptions',
                'switch_groups',
                'switch_genders',
                'switch_customers',
                'switch_categories',
                'switch_manufacturers',
                'switch_suppliers',
                'switch_products',
                'switch_countries',
                'switch_zones',
                'switch_cms',
                'switch_categories_selected',
                'switch_languages',
                'switch_attributes',
                'switch_features',
                'switch_display_url_string',
                'switch_display_referrer_string',
                'switch_display_ip_string',
            ];

            // Reset fields with selector to NO but selected values remain
            foreach ($switchFields as $switchField) {
                if (!(int) Tools::getValue($switchField)) {
                    // Set value in $_POST, can't use Tools::getValue()
                    if ($switchField === 'switch_display_url_string') {
                        foreach (Language::getLanguages(false) as $language) {
                            $_POST['display_url_string_' . $language['id_lang']] = '';
                        }
                    } elseif ($switchField === 'switch_display_referrer_string') {
                        foreach (Language::getLanguages(false) as $language) {
                            $_POST['display_referrer_string_' . $language['id_lang']] = '';
                        }
                    } elseif ($switchField === 'switch_display_ip_string') {
                        $_POST['display_ip_string'] = '';
                    } else {
                        unset($_POST[str_replace('switch_', '', $switchField)]);
                        if ($switchField === 'switch_categories_selected') {
                            $_POST['nb_products'] = '';
                        }
                    }
                }
            }
        }

        if (Tools::getValue('deleteImage') && Tools::getValue('id_language') && Tools::getValue('type')) {
            if (($object = $this->loadObject(true)) || Validate::isLoadedObject($object)) {
                $type = Tools::getValue('type');
                $image = $object->{$type}[(int) Tools::getValue('id_language')];
                if ($this->module->deleteImage(_PS_MODULE_DIR_ . AdvancedPopupCreator::$image_dir . $image)) {
                    $object->{$type}[(int) Tools::getValue('id_language')] = '';
                    $object->save();
                    Tools::redirectAdmin(self::$currentIndex . '&add' . $this->table . '&' . $this->identifier . '=' . Tools::getValue($this->identifier) . '&conf=7&token=' . Tools::getAdminTokenLite($this->tabClassName));
                }
            }
            $this->errors[] = $this->module->l('An error occurred during image deletion (cannot load object).');
        }

        parent::postProcess();

        if (!empty($this->errors)) {
            // if we have errors, we stay on the form instead of going back to the list
            $this->display = 'edit';

            return false;
        }
    }

    public function renderList()
    {
        // Redirect if no popup created
        if (!$this->errors && $this->isShopSelected && !AdvancedPopup::getNbObjects()) {
            $this->redirect_after = 'index.php?controller=' . $this->tabClassName . '&add' . $this->table . '&token=' . Tools::getAdminTokenLite($this->tabClassName);
            $this->redirect();
        }

        if ($this->isShopSelected
            && (
                (version_compare(_PS_VERSION_, '1.5.0.13', '<')
                    && !Module::isInstalled($this->module->name))
             || (version_compare(_PS_VERSION_, '1.5.0.13', '>=')
                    && !Module::isEnabled($this->module->name))
            )
        ) {
            $this->warnings[] = $this->module->l('Module is not enabled in this shop.');
        }

        $this->addRowAction('duplicate');

        return parent::renderList();
    }

    public function renderForm()
    {
        if (!$this->isShopSelected && $this->display == 'add') {
            $this->errors[] = $this->module->l('You have to select a shop if you want to create a new popup.');

            return;
        }

        if ($this->isShopSelected
            && (
                (version_compare(_PS_VERSION_, '1.5.0.13', '<')
                    && !Module::isInstalled($this->module->name))
             || (version_compare(_PS_VERSION_, '1.5.0.13', '>=')
                    && !Module::isEnabled($this->module->name))
            )
        ) {
            $this->warnings[] = $this->module->l('Module is not enabled in this shop.');
        }

        if (!($object = $this->loadObject(true))) {
            return;
        }

        // Controller exceptions
        $controllers = Dispatcher::getControllers(_PS_FRONT_CONTROLLER_DIR_);
        $module_controllers = $this->getModuleControllers();
        $exceptions_controllers = array_merge($controllers, $module_controllers);
        $list_controllers = $this->getControllersExceptions($exceptions_controllers);

        // Get lists data
        if (version_compare(_PS_VERSION_, '1.5', '<')) {
            $id_lang = (int) $this->context->cookie->id_lang;
        } else {
            $id_lang = (int) $this->context->language->id;
        }

        $groups = Group::getGroups($id_lang, true);

        $categories = Category::getCategories($id_lang, false, false, 'AND c.`level_depth` > 1', 'ORDER BY cl.`name` ASC');
        foreach ($categories as &$category) {
            $category['display_name'] = $category['name'] . ' (ID: ' . $category['id_category'] . ')';
        }
        $categories_selected = $categories;

        $countries = Country::getCountries($id_lang);

        $this->multiple_fieldsets = true;
        $this->default_form_language = $this->context->language->id;

        $this->context->smarty->assign([
            'products_selected' => $object->products,
            'products_available' => '', // $products,
            'object' => $object,
            'image_field' => 'image',
            'image_background_field' => 'image_background',
            'languages' => Language::getLanguages(false),
            'default_form_language' => $this->default_form_language,
            'image_dir' => _MODULE_DIR_ . AdvancedPopupCreator::$image_dir,
            'images' => $object->image,
            'images_background' => $object->image_background,
            'delete_url' => self::$currentIndex . '&' . $this->identifier . '=' . $object->id_advancedpopup . '&token=' . Tools::getAdminTokenLite($this->tabClassName) . '&deleteImage=1',
            'schedule' => $object->schedule,
            'objectId' => $object->id,
        ]);

        $fieldsFormIndex = 0;
        $this->fields_form[$fieldsFormIndex]['form'] = [
            'legend' => [
                'title' => $this->module->l('Popup configuration'),
                'icon' => 'icon-wrench',
            ],
            'input' => [
                [
                    'type' => 'text',
                    'label' => $this->module->l('Name'),
                    'name' => 'name',
                    'required' => true,
                    'col' => '6',
                    'desc' => $this->module->l('Descriptive name to identify this popup'),
                ],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l('Active'),
                    'name' => 'active',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'active_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'active_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                ],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'datetime' : 'date',
                    'label' => $this->module->l('From'),
                    'name' => 'date_init',
                    'class' => 'datetimepicker fixed-width-xl',
                    'desc' => $this->module->l('Date when the popup will first appear on the site'),
                ],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'datetime' : 'date',
                    'label' => $this->module->l('To'),
                    'name' => 'date_end',
                    'class' => 'datetimepicker fixed-width-xl',
                    'desc' => $this->module->l('Date when the popup will stop being displayed'),
                ],
                [
                    'type' => 'html',
                    'label' => $this->module->l('Viewing schedule'),
                    'name' => $this->context->smarty->fetch($this->module->getLocalPath() . 'views/templates/admin/schedule.tpl'),
                    'desc' => $this->module->l('Select the days of the week and time range during which the popup will be visible. Click on each day to enable or disable it, then define the specific hours for display'),
                ],
                /*[
                    'type' => 'text',
                    'label' => $this->module->l('Priority'),
                    'name' => 'priority',
                    'class' => 'fixed-width-sm',
                    'required' => true,
                    'col' => 8,
                ],*/
            ],
            'submit' => [
                'title' => $this->module->l('Save'),
                'type' => 'submit',
            ],
            'buttons' => [
                'save-and-stay' => [
                    'title' => $this->module->l('Save and stay'),
                    'name' => 'submitAdd' . $this->table . 'AndStay',
                    'type' => 'submit',
                    'class' => 'btn btn-default pull-right',
                    'icon' => 'process-icon-save',
                ],
            ],
        ];

        ++$fieldsFormIndex;
        $this->fields_form[$fieldsFormIndex]['form'] = [
            'legend' => [
                'title' => $this->module->l('Popup triggers'),
                'icon' => 'icon-fire',
            ],
            'input' => [
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l('Display on Page Load'),
                    'name' => 'display_on_load',
                    'class' => 't',
                    'col' => '8',
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'display_on_load_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'display_on_load_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Shows the popup immediately after the page finishes loading.'),
                ],
                [
                    'type' => 'text',
                    'label' => $this->module->l('Delay Popup Display After Page Load'),
                    'name' => 'secs_to_display',
                    'class' => 'fixed-width-sm',
                    'col' => '8',
                    'suffix' => $this->module->l('seconds'),
                    'desc' => $this->module->l('Wait for a specified number of seconds after the page has loaded before showing the popup.') . '<br />' . $this->module->l('Leave empty if you want the popup to display immediately after the page loads.'),
                ],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l('Display After Product Added to Cart'),
                    'name' => 'display_after_cart',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'display_after_cart_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'display_after_cart_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Shows the popup once a product is added to the cart.'),
                ],
                [
                    'type' => 'text',
                    'label' => $this->module->l('Delay Popup Display After Product Added to Cart'),
                    'name' => 'secs_to_display_cart',
                    'class' => 'fixed-width-sm',
                    'col' => '8',
                    'suffix' => $this->module->l('seconds'),
                    'desc' => $this->module->l('Wait for a specified number of seconds after adding a product to the cart before showing the popup.') . '<br />' . $this->module->l('Leave empty if you want the popup to display immediately after the page loads.'),
                ],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l('Exit Intent Popup'),
                    'name' => 'display_on_exit',
                    'class' => 't',
                    'col' => '8',
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'display_on_exit_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'display_on_exit_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Displays the popup when the user attempts to leave the page (detected by mouse movement towards the top of the screen).') . '<br />' . $this->module->l('Please note, this feature is not supported on touch devices.'),
                ],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l('Display on User Inactivity'),
                    'name' => 'display_inactive',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'display_inactive_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'display_inactive_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Shows the popup after the user has been inactive for a certain period of time.'),
                ],
                [
                    'type' => 'text',
                    'label' => $this->module->l('Delay Popup After Inactivity'),
                    'name' => 'secs_to_display_inactive',
                    'class' => 'fixed-width-sm',
                    'col' => '8',
                    'suffix' => $this->module->l('seconds'),
                    'desc' => $this->module->l('Wait for a specified number of seconds of user inactivity before displaying the popup.') . '<br />' . $this->module->l('Leave empty if you want the popup to display immediately after the page loads.'),
                ],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l('Display on User Click on an Element'),
                    'name' => 'display_on_click',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'display_on_click_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'display_on_click_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Shows the popup when the user clicks on a specific element on the page.'),
                ],
                [
                    'type' => 'text',
                    'label' => $this->module->l('jQuery Element Selector'),
                    'name' => 'display_on_click_selector',
                    'class' => 'fixed-width-xxl',
                    'col' => '8',
                    'desc' => $this->module->l('Specify a jQuery selector for the element that triggers the popup when clicked (e.g., a button or link).') . '</br>' . $this->module->l('Example: $("footer .header1")') . '</br>' . $this->module->l('You can use extension "jQuery Unique Selector" to get the correct jQuery selector'),
                ],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l('Display on Click on Sticky Block'),
                    'name' => 'display_on_sticky',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'display_on_sticky_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'display_on_sticky_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Shows the popup when the user clicks on the sticky block.'),
                ],
                [
                    'type' => 'html',
                    'name' => 'dummy',
                    'html_content' => '<hr>',
                ],
                [
                    'type' => 'text',
                    'label' => $this->module->l('Display After Viewing X Pages'),
                    'name' => 'show_on_view_page_nbr',
                    'col' => '8',
                    'class' => 'fixed-width-sm',
                    'desc' => $this->module->l('Displays the popup after the customer has viewed the specified number of pages.') . '<br />' . $this->module->l('Set to 0 if you want the popup to appear on the first page view.'),
                ],
                [
                    'type' => 'text',
                    'label' => $this->module->l('Re-display After X Minutes'),
                    'name' => 'show_every_nbr_hours',
                    'class' => 'fixed-width-sm',
                    'col' => '8',
                    'suffix' => $this->module->l('minutes'),
                    'desc' => $this->module->l('Sets a delay in minutes before the popup can appear again after being dismissed.') . '</br>' . $this->module->l('Leave empty to Display the popup every time the page loads or the event is triggered.') . '</br>' . $this->module->l('Set "99999" to display the popup only once.'),
                ],

                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l('Reset Timer Per Session'),
                    'name' => 'session',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'session_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'session_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Resets the popup display timer at the beginning of each session, allowing the popup to be triggered based on session activity or page views.'),
                ],
                [
                    'type' => 'text',
                    'label' => $this->module->l('Maximum Views per User'),
                    'name' => 'maximum_per_user',
                    'class' => 'fixed-width-sm',
                    'col' => '8',
                    'suffix' => $this->module->l('times'),
                    'desc' => $this->module->l('Limits the number of times the popup can be shown to a user during a session or across multiple visits.'),
                ],
                [
                    'type' => 'html',
                    'name' => 'dummy',
                    'html_content' => '<hr>',
                ],
                [
                    'type' => 'text',
                    'label' => $this->module->l('Auto-Close Popup After X Seconds'),
                    'name' => 'secs_to_close',
                    'class' => 'fixed-width-sm',
                    'col' => 8,
                    'suffix' => $this->module->l('seconds'),
                    'desc' => $this->module->l('Automatically closes the popup after the specified number of seconds.') . '</br>' . $this->module->l('Leave empty if you don’t want it to close automatically.'),
                ],
            ],
            'submit' => [
                'title' => $this->module->l('Save'),
                'type' => 'submit',
            ],
            'buttons' => [
                'save-and-stay' => [
                    'title' => $this->module->l('Save and stay'),
                    'name' => 'submitAdd' . $this->table . 'AndStay',
                    'type' => 'submit',
                    'class' => 'btn btn-default pull-right',
                    'icon' => 'process-icon-save',
                ],
            ],
        ];

        ++$fieldsFormIndex;
        $this->fields_form[$fieldsFormIndex]['form'] = [
            'legend' => [
                'title' => $this->module->l('Popup content'),
                'icon' => 'icon-magic',
            ],
            'input' => [
                [
                    'type' => 'textarea',
                    'label' => $this->module->l('Popup content'),
                    'name' => 'content',
                    'id' => 'html_content',
                    'lang' => true,
                    'autoload_rte' => version_compare(_PS_VERSION_, '1.6', '>=') ? '' : true,
                    'cols' => 120,
                    'rows' => 10,
                    'class' => version_compare(_PS_VERSION_, '1.6', '>=') ? 'apc_tiny' : '',
                    'desc' => $this->module->l('Enter the content that will be displayed in the popup (e.g., text, HTML, or other media).'),
                ],
                [
                    'type' => 'color',
                    'label' => $this->module->l('Background color'),
                    'name' => 'color_background',
                    'col' => '6',
                    'size' => '100',
                    'desc' => $this->module->l('Choose a background color for the popup. This color will cover the entire popup area.'),
                ],
                [
                    'type' => 'html',
                    'label' => $this->module->l('Background image'),
                    'name' => $this->context->smarty->fetch(_PS_MODULE_DIR_ . 'advancedpopupcreator/views/templates/admin/image-background.tpl'),
                    'desc' => $this->module->l('Upload an image to be used as the background for the popup. It will be displayed behind the content.'),
                ],
                [
                    'type' => 'html',
                    'label' => '', // Empty label to fix offset
                    'name' => $this->context->smarty->fetch(_PS_MODULE_DIR_ . 'advancedpopupcreator/views/templates/admin/image-background-preview.tpl'),
                ],
                [
                    'type' => 'html',
                    'label' => $this->module->l('Content image'),
                    'name' => $this->context->smarty->fetch(_PS_MODULE_DIR_ . 'advancedpopupcreator/views/templates/admin/image.tpl'),
                    'desc' => $this->module->l('Upload an image to be displayed within the popup content area. This image will be shown alongside or inside the text.'),
                ],
                [
                    'type' => 'html',
                    'label' => '', // Empty label to fix offset
                    'name' => $this->context->smarty->fetch(_PS_MODULE_DIR_ . 'advancedpopupcreator/views/templates/admin/image-preview.tpl'),
                ],
                [
                    'type' => 'text',
                    'label' => $this->module->l('Link for the Image'),
                    'name' => 'image_link',
                    'lang' => true,
                    'col' => '8',
                    'desc' => $this->module->l('Specify a URL that the image will link to when clicked.'),
                ],
                [
                    'type' => 'select',
                    'label' => $this->module->l('Target for Image Link'),
                    'name' => 'image_link_target',
                    'class' => 't',
                    'col' => 8,
                    'options' => [
                        'query' => [
                            [
                                'id' => '_self',
                                'name' => $this->module->l('In the same page'),
                            ],
                            [
                                'id' => '_blank',
                                'name' => $this->module->l('In a new page'),
                            ],
                        ],
                        'id' => 'id',
                        'name' => 'name',
                    ],
                    'desc' => $this->module->l('Choose where the link should open: in the same window or a new tab.'),
                ],
            ],
            'submit' => [
                'title' => $this->module->l('Save'),
                'type' => 'submit',
            ],
            'buttons' => [
                'save-and-stay' => [
                    'title' => $this->module->l('Save and stay'),
                    'name' => 'submitAdd' . $this->table . 'AndStay',
                    'type' => 'submit',
                    'class' => 'btn btn-default pull-right',
                    'icon' => 'process-icon-save',
                ],
            ],
        ];

        ++$fieldsFormIndex;
        $this->fields_form[$fieldsFormIndex]['form'] = [
            'legend' => [
                'title' => $this->module->l('Appearance'),
                'icon' => 'icon-camera-retro',
            ],
            'input' => [
                [
                    'type' => 'select',
                    'label' => $this->module->l('Position'),
                    'name' => 'position',
                    'class' => 't',
                    'col' => 8,
                    'options' => [
                        'query' => [
                            [
                                'id' => '1',
                                'name' => $this->module->l('Top left'),
                            ],
                            [
                                'id' => '2',
                                'name' => $this->module->l('Top center'),
                            ],
                            [
                                'id' => '3',
                                'name' => $this->module->l('Top right'),
                            ],
                            [
                                'id' => '4',
                                'name' => $this->module->l('Center left'),
                            ],
                            [
                                'id' => '5',
                                'name' => $this->module->l('Center'),
                            ],
                            [
                                'id' => '6',
                                'name' => $this->module->l('Center right'),
                            ],
                            [
                                'id' => '7',
                                'name' => $this->module->l('Bottom left'),
                            ],
                            [
                                'id' => '8',
                                'name' => $this->module->l('Bottom center'),
                            ],
                            [
                                'id' => '9',
                                'name' => $this->module->l('Bottom right'),
                            ],
                        ],
                        'id' => 'id',
                        'name' => 'name',
                    ],
                    'desc' => $this->module->l('Choose where on the screen the popup will appear.'),
                ],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l('Show "Don\'t Show Again" Option'),
                    'name' => 'dont_display_again',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'dont_display_again_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'dont_display_again_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Allow users to opt-out of seeing the popup again by providing a "Don\'t show this message again" checkbox.'),
                ],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l('Blur background'),
                    'name' => 'blur_background',
                    'class' => 't',
                    'col' => '8',
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'blur_background_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'blur_background_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Apply a blur effect to the background when the popup is displayed. Use this to draw attention to the popup content.'),
                ],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l('Show Backdrop Overlay'),
                    'name' => 'display_backdrop',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'display_backdrop_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'display_backdrop_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Enable a semi-transparent overlay behind the popup to focus attention on the modal window.'),
                ],
                [
                    'type' => 'select',
                    'label' => $this->module->l('Backdrop Opacity'),
                    'name' => 'back_opacity_value',
                    'class' => 't',
                    'col' => 8,
                    'options' => [
                        'query' => [
                            [
                                'id' => '1',
                                'name' => '1',
                            ],
                            [
                                'id' => '0.9',
                                'name' => '0.9',
                            ],
                            [
                                'id' => '0.8',
                                'name' => '0.8',
                            ],
                            [
                                'id' => '0.7',
                                'name' => '0.7',
                            ],
                            [
                                'id' => '0.6',
                                'name' => '0.6',
                            ],
                            [
                                'id' => '0.5',
                                'name' => '0.5',
                            ],
                            [
                                'id' => '0.4',
                                'name' => '0.4',
                            ],
                            [
                                'id' => '0.3',
                                'name' => '0.3',
                            ],
                            [
                                'id' => '0.2',
                                'name' => '0.2',
                            ],
                            [
                                'id' => '0.1',
                                'name' => '0.1',
                            ],
                        ],
                        'id' => 'id',
                        'name' => 'name',
                    ],
                    'desc' => $this->module->l('Adjust the opacity of the backdrop overlay. Lower values make the background more transparent.'),
                ],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l('Close Popup on Background Click'),
                    'name' => 'close_on_background',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'close_on_background_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'close_on_background_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Allow the popup to be closed when the user clicks on the backdrop outside the popup window.'),
                ],
                [
                    'type' => 'html',
                    'lang' => true,
                    'col' => '8',
                    'label' => $this->module->l('Popup height'),
                    'name' => $this->context->smarty->fetch(_PS_MODULE_DIR_ . 'advancedpopupcreator/views/templates/admin/popup_height.tpl'),
                    'desc' => $this->module->l('Define the height of the popup. Leave empty for automatic height adjustment based on content.'),
                ],
                [
                    'type' => 'html',
                    'lang' => true,
                    'col' => '8',
                    'label' => $this->module->l('Popup width'),
                    'name' => $this->context->smarty->fetch(_PS_MODULE_DIR_ . 'advancedpopupcreator/views/templates/admin/popup_width.tpl'),
                    'desc' => $this->module->l('Set the width of the popup. Leave empty for automatic width adjustment based on content.'),
                ],
                [
                    'type' => 'text',
                    'label' => $this->module->l('Popup padding'),
                    'name' => 'popup_padding',
                    'suffix' => 'px',
                    'lang' => true,
                    'col' => '8',
                    'class' => 'fixed-width-sm',
                    'desc' => $this->module->l('Define the padding around the popup content in pixels to control the space between the content and the popup edges.'),
                ],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l('Enable Scroll on Content Page'),
                    'name' => 'locked',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'locked_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'locked_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Allow the page content to be scrollable behind the popup.'),
                ],
                /*[
                    'type' => 'select',
                    'label' => $this->module->l('Open effect'),
                    'name' => 'open_effect',
                    'class' => 't',
                    'col' => 8,
                    'options' => [
                        'query' => [
                            [
                                'id' => 'none',
                                'name' => $this->module->l('None'),
                            ],
                            [
                                'id' => 'bounce',
                                'name' => $this->module->l('bounce'),
                            ],
                            [
                                'id' => 'bounceIn',
                                'name' => $this->module->l('bounceIn'),
                            ],
                            [
                                'id' => 'bounceUpIn',
                                'name' => $this->module->l('bounceUpIn'),
                            ],
                            [
                                'id' => 'slideright',
                                'name' => $this->module->l('Slide in from the right'),
                            ],
                            [
                                'id' => 'slidebottom',
                                'name' => $this->module->l('Slide in from the bottom'),
                            ],
                            [
                                'id' => 'newspaper',
                                'name' => $this->module->l('Newspaper (twirl in)'),
                            ],
                            [
                                'id' => 'fall',
                                'name' => $this->module->l('Fall'),
                            ],
                            [
                                'id' => 'sidefall',
                                'name' => $this->module->l('Fall from the side'),
                            ],
                            [
                                'id' => 'stickyup',
                                'name' => $this->module->l('Slide from the top'),
                            ],
                            [
                                'id' => 'horizontalflip',
                                'name' => $this->module->l('Horizontal 3D flip'),
                            ],
                            [
                                'id' => 'verticalflip',
                                'name' => $this->module->l('Vertical 3D flip'),
                            ],
                            [
                                'id' => 'sign',
                                'name' => $this->module->l('3D swinging sign'),
                            ],
                            [
                                'id' => 'slit',
                                'name' => $this->module->l('3D growing slit'),
                            ],
                            [
                                'id' => 'rotatebottom',
                                'name' => $this->module->l('3D rotate from bottom'),
                            ],
                            [
                                'id' => 'rotateleft',
                                'name' => $this->module->l('3D rotate from left'),
                            ],
                        ],
                        'id' => 'id',
                        'name' => 'name',
                    ],
                ],*/
                [
                    'type' => 'text',
                    'label' => $this->module->l('Add Custom CSS Class to Popup Layout'),
                    'name' => 'css_class',
                    'lang' => true,
                    'col' => '8',
                    'class' => 'fixed-width-xxl',
                    'desc' => $this->module->l('Add a custom CSS class to the popup for easy styling.'),
                ],
                [
                    'type' => 'textarea',
                    'label' => $this->module->l('Custom CSS Rules'),
                    'name' => 'css',
                    'class' => 't codemirror codemirror-css',
                    'lang' => true,
                    'cols' => 100,
                    'rows' => 10,
                    'desc' => $this->module->l('Enter custom CSS rules to style the popup as needed, providing more control over its appearance.'),
                ],
                [
                    'type' => 'textarea',
                    'label' => $this->module->l('Custom JavaScript Code'),
                    'name' => 'js',
                    'class' => 't codemirror codemirror-js',
                    'lang' => true,
                    'cols' => 100,
                    'rows' => 10,
                    'desc' => $this->module->l('Add custom JavaScript code to run specific actions or interactions within the popup.'),
                ],
            ],
            'submit' => [
                'title' => $this->module->l('Save'),
                'type' => 'submit',
            ],
            'buttons' => [
                'save-and-stay' => [
                    'title' => $this->module->l('Save and stay'),
                    'name' => 'submitAdd' . $this->table . 'AndStay',
                    'type' => 'submit',
                    'class' => 'btn btn-default pull-right',
                    'icon' => 'process-icon-save',
                ],
            ],
        ];

        ++$fieldsFormIndex;
        $this->fields_form[$fieldsFormIndex]['form'] = [
            'legend' => [
                'title' => $this->module->l('Who to display popup'),
                'icon' => 'icon-check',
            ],
            'buttons' => [
                'save-and-stay' => [
                    'title' => $this->module->l('Save and stay'),
                    'name' => 'submitAdd' . $this->table . 'AndStay',
                    'type' => 'submit',
                    'class' => 'btn btn-default pull-right',
                    'icon' => 'process-icon-save',
                ],
            ],
            'input' => [
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l('Show on Mobile Devices'),
                    'name' => 'display_mobile',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'display_mobile_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'display_mobile_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Enable the popup to appear on mobile devices.'),
                ],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l('Show on Tablet Devices'),
                    'name' => 'display_tablet',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'display_tablet_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'display_tablet_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Enable the popup to appear on tablet devices.'),
                ],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l(' Show on Desktop Devices'),
                    'name' => 'display_desktop',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'display_desktop_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'display_desktop_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Enable the popup to appear on desktop devices.'),
                ],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l('Show Only to Newsletter Subscribers'),
                    'name' => 'show_customer_newsletter',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'show_customer_newsletter_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'show_customer_newsletter_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Display the popup only to customers who are subscribed to the newsletter.'),
                ],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l('Show Only to Non-Subscribers'),
                    'name' => 'show_customer_not_newsletter',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'show_customer_not_newsletter_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'show_customer_not_newsletter_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Display the popup only to customers who are not subscribed to the newsletter.'),
                ],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l('Show Only if URL Contains Specific String'),
                    'name' => 'switch_display_url_string',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'display_url_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'display_url_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Display the popup only when a specific string is present in the page URL.'),
                ],
                [
                    'type' => 'text',
                    'label' => $this->module->l('URL String'),
                    'name' => 'display_url_string',
                    'class' => 'switch_display_url_string',
                    'lang' => true,
                    'col' => 8,
                    'desc' => $this->module->l('Enter the string that must appear in the URL for the popup to be displayed.'),
                ],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l('Show Only for Specific Referrers'),
                    'name' => 'switch_display_referrer_string',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'display_referrer_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'display_referrer_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Display the popup only if the customer is referred from a specific URL.'),
                ],
                [
                    'type' => 'text',
                    'label' => $this->module->l('Referrer URL'),
                    'name' => 'display_referrer_string',
                    'class' => 'switch_display_referrer_string',
                    'lang' => true,
                    'col' => 8,
                    'desc' => $this->module->l('Enter the referrer URL that will trigger the popup display when the customer comes from it.'),
                ],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l('Show Only to Specific IPs'),
                    'name' => 'switch_display_ip_string',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'display_ip_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'display_ip_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Limit the popup display to customers with specific IP addresses.'),
                ],
                [
                    'type' => 'text',
                    'label' => $this->module->l('Enter IPs (Comma-Separated)'),
                    'name' => 'display_ip_string',
                    'class' => 'switch_display_ip_string',
                    'col' => 8,
                    'desc' => $this->module->l('Enter the IP addresses, separated by commas, to which the popup will be shown.'),
                ],
            ],
        ];

        $gendersCollection = Gender::getGenders((int) $this->context->language->id);
        $genders = [];
        if ($gendersCollection->count() > 1) {
            foreach ($gendersCollection as $gendersKey => $gender) {
                $genders[$gendersKey]['id_gender'] = $gender->id;
                $genders[$gendersKey]['name'] = $gender->name;
            }

            $genderFieldSwitch = [
                'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                'label' => sprintf('%1$s %2$s', $this->module->l('Filter by'), $this->module->l('Gender')),
                'name' => 'switch_genders',
                'class' => 't',
                'col' => 8,
                'is_bool' => true,
                'values' => [
                    [
                        'id' => 'switch_genders_on',
                        'value' => 1,
                        'label' => $this->module->l('Enabled'),
                    ],
                    [
                        'id' => 'switch_genders_off',
                        'value' => 0,
                        'label' => $this->module->l('Disabled'),
                    ],
                ],
                'desc' => $this->module->l('Display the popup only to customers of the selected gender(s).'),
            ];
            $genderField = [
                'type' => 'swap-custom',
                'label' => $this->module->l('Show Only to Selected Genders'),
                'name' => 'genders[]',
                'class' => 'switch_genders',
                'search' => true,
                'options' => [
                    'query' => $genders,
                    'id' => 'id_gender',
                    'name' => 'name',
                ],
                'desc' => $this->module->l('Choose which genders will trigger the popup to appear.'),
            ];

            array_push($this->fields_form[$fieldsFormIndex]['form']['input'], $genderFieldSwitch, $genderField);
        }

        if (!Configuration::get('APC_HIDE_FILTER_CUSTOMER')) {
            if ($this->getNbCustomers()) {
                $customers = Customer::getCustomers(true);
                $customerFieldSwitch = [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => sprintf('%1$s %2$s', $this->module->l('Filter by'), $this->module->l('Customer')),
                    'name' => 'switch_customers',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'switch_customers_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'switch_customers_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Display the popup only to selected customers.'),
                ];
                $customerField = [
                    'type' => 'swap-custom',
                    'label' => $this->module->l('Show Only to Selected Customers'),
                    'name' => 'customers[]',
                    'class' => 'switch_customers',
                    'search' => true,
                    'options' => [
                        'query' => $customers,
                        'id' => 'id_customer',
                        'name' => 'email',
                    ],
                    'desc' => $this->module->l('Select the customers who should see the popup.'),
                ];

                array_push($this->fields_form[$fieldsFormIndex]['form']['input'], $customerFieldSwitch, $customerField);
            }
        }

        if (!Configuration::get('APC_HIDE_FILTER_CUSTOMER_GROUP')) {
            if (count($groups) && Group::isFeatureActive()) {
                $groupFieldSwitch = [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => sprintf('%1$s %2$s', $this->module->l('Filter by'), $this->module->l('Customer group')),
                    'name' => 'switch_groups',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'switch_groups_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'switch_groups_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Display the popup only to customers belonging to the selected group(s).'),
                ];

                $groupField = [
                    'type' => 'swap-custom',
                    'label' => $this->module->l('Show Only to Selected Customer Groups'),
                    'name' => 'groups[]',
                    'class' => 'switch_groups',
                    'search' => true,
                    'options' => [
                        'query' => $groups,
                        'id' => 'id_group',
                        'name' => 'name',
                    ],
                    'desc' => $this->module->l('Choose the customer groups to which the popup should be shown.'),
                ];

                array_push($this->fields_form[$fieldsFormIndex]['form']['input'], $groupFieldSwitch, $groupField);
            }
        }

        array_push(
            $this->fields_form[$fieldsFormIndex]['form']['input'],
            [
                'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                'label' => sprintf('%1$s %2$s', $this->module->l('Filter by'), $this->module->l('Country')),
                'name' => 'switch_countries',
                'class' => 't',
                'col' => 8,
                'is_bool' => true,
                'values' => [
                    [
                        'id' => 'switch_countries_on',
                        'value' => 1,
                        'label' => $this->module->l('Enabled'),
                    ],
                    [
                        'id' => 'switch_countries_off',
                        'value' => 0,
                        'label' => $this->module->l('Disabled'),
                    ],
                ],
                'desc' => $this->module->l('Display the popup only in the selected countries.'),
            ],
            [
                'type' => 'swap-custom',
                'label' => $this->module->l('Show Only to Selected Countries'),
                'name' => 'countries[]',
                'class' => 'switch_countries',
                'search' => true,
                'options' => [
                    'query' => $countries,
                    'id' => 'id_country',
                    'name' => 'name',
                ],
                'desc' => $this->module->l('Select the countries where the popup should be displayed.'),
            ],
            [
                'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                'label' => sprintf('%1$s %2$s', $this->module->l('Filter by'), $this->module->l('Zone')),
                'name' => 'switch_zones',
                'class' => 't',
                'col' => 8,
                'is_bool' => true,
                'values' => [
                    [
                        'id' => 'switch_zones_on',
                        'value' => 1,
                        'label' => $this->module->l('Enabled'),
                    ],
                    [
                        'id' => 'switch_zones_off',
                        'value' => 0,
                        'label' => $this->module->l('Disabled'),
                    ],
                ],
                'desc' => $this->module->l('Display the popup only in the selected geographic zones.'),
            ],
            [
                'type' => 'swap-custom',
                'label' => $this->module->l('Show Only to Selected Zones'),
                'name' => 'zones[]',
                'class' => 'switch_zones',
                'search' => true,
                'options' => [
                    'query' => Zone::getZones(),
                    'id' => 'id_zone',
                    'name' => 'name',
                ],
                'desc' => $this->module->l('Choose the zones where the popup will be displayed.'),
            ]
        );

        array_push(
            $this->fields_form[$fieldsFormIndex]['form']['input'],
            [
                'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                'label' => sprintf('%1$s %2$s', $this->module->l('Filter by'), $this->module->l('Cart amount')),
                'name' => 'cart_amount',
                'class' => 't',
                'col' => 8,
                'is_bool' => true,
                'values' => [
                    [
                        'id' => 'cart_amount_on',
                        'value' => 1,
                        'label' => $this->module->l('Enabled'),
                    ],
                    [
                        'id' => 'cart_amount_off',
                        'value' => 0,
                        'label' => $this->module->l('Disabled'),
                    ],
                ],
                'desc' => $this->module->l('Display the popup only if the cart\'s total value matches the specified amount.'),
            ],
            [
                'type' => 'text',
                'label' => $this->module->l('From'),
                'name' => 'cart_amount_from',
                'class' => 'fixed-width-md',
                'prefix' => '>=',
                'suffix' => version_compare(_PS_VERSION_, '1.7.6', '>=') ? $this->context->currency->symbol : $this->context->currency->sign,
                'col' => 8,
                'desc' => $this->module->l('Set the minimum cart value required for the popup to be displayed.'),
            ],
            [
                'type' => 'text',
                'label' => $this->module->l('To'),
                'name' => 'cart_amount_to',
                'class' => 'fixed-width-md',
                'prefix' => '<=',
                'suffix' => version_compare(_PS_VERSION_, '1.7.6', '>=') ? $this->context->currency->symbol : $this->context->currency->sign,
                'col' => 8,
                'desc' => $this->module->l('Set the maximum cart value required for the popup to be displayed.'),
            ]
        );

        if (!Configuration::get('APC_HIDE_FILTER_CATEGORY')) {
            array_push(
                $this->fields_form[$fieldsFormIndex]['form']['input'],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => sprintf('%1$s %2$s', $this->module->l('Filter by'), $this->module->l('Products in the cart')),
                    'name' => 'switch_categories_selected',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'switch_categories_selected_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'switch_categories_selected_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Display the popup only if the cart contains the selected products.'),
                ],
                [
                    'type' => 'html',
                    'label' => $this->module->l('Show Only if Cart Contains Products from Selected Categories'),
                    'name' => $this->context->smarty->fetch(_PS_MODULE_DIR_ . 'advancedpopupcreator/views/templates/admin/nb_products_template.tpl'),
                    'desc' => $this->module->l('Display the popup only if the cart contains a specified number of products from the selected categories.'),
                ],
                [
                    'type' => 'swap-custom',
                    'label' => '',
                    'class' => 'switch_categories_selected',
                    'name' => 'categories_selected[]',
                    'search' => true,
                    'options' => [
                        'query' => $categories_selected,
                        'id' => 'id_category',
                        'name' => 'display_name',
                    ],
                ]
            );
        }

        array_push(
            $this->fields_form[$fieldsFormIndex]['form']['input'],
            [
                'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                'label' => sprintf('%1$s %2$s', $this->module->l('Filter by'), $this->module->l('Language')),
                'name' => 'switch_languages',
                'class' => 't',
                'col' => 8,
                'is_bool' => true,
                'values' => [
                    [
                        'id' => 'switch_languages_on',
                        'value' => 1,
                        'label' => $this->module->l('Enabled'),
                    ],
                    [
                        'id' => 'switch_languages_off',
                        'value' => 0,
                        'label' => $this->module->l('Disabled'),
                    ],
                ],
                'desc' => $this->module->l('Display the popup only in the selected languages.'),
            ],
            [
                'type' => 'swap-custom',
                'label' => $this->module->l('Show Only to Selected Languages'),
                'name' => 'languages[]',
                'class' => 'switch_languages',
                'search' => true,
                'options' => [
                    'query' => Language::getLanguages(false),
                    'id' => 'id_lang',
                    'name' => 'name',
                ],
                'desc' => $this->module->l('Choose the languages in which the popup will be shown.'),
            ],
            [
                'type' => 'text',
                'label' => $this->module->l('Show Only if Screen Width Exceeds X px'),
                'name' => 'responsive_min',
                'class' => 'fixed-width-md',
                'lang' => true,
                'col' => 8,
                'suffix' => $this->module->l('px'),
                'desc' => $this->module->l('Display the popup only if the screen width is greater than the specified number of pixels.'),
            ],
            [
                'type' => 'text',
                'label' => $this->module->l('Show Only if Screen Width is Less Than X px'),
                'name' => 'responsive_max',
                'class' => 'fixed-width-md',
                'lang' => true,
                'col' => 8,
                'suffix' => $this->module->l('px'),
                'desc' => $this->module->l('Display the popup only if the screen width is smaller than the specified number of pixels.'),
            ]
        );

        $this->fields_form[$fieldsFormIndex]['form']['submit'] = [
            'title' => $this->module->l('Save'),
            'type' => 'submit',
        ];

        ++$fieldsFormIndex;
        $this->fields_form[$fieldsFormIndex]['form'] = [
            'legend' => [
                'title' => $this->module->l('Where to display popup'),
                'icon' => 'icon-globe',
            ],
            'buttons' => [
                'save-and-stay' => [
                    'title' => $this->module->l('Save and stay'),
                    'name' => 'submitAdd' . $this->table . 'AndStay',
                    'type' => 'submit',
                    'class' => 'btn btn-default pull-right',
                    'icon' => 'process-icon-save',
                ],
            ],
            'input' => [],
        ];

        if (!Configuration::get('APC_HIDE_FILTER_PRODUCT')) {
            if ($this->getNbProducts()) {
                $products = $this->getProductsLite($id_lang, true, true);
                foreach ($products as &$product) {
                    $product['display_name'] = $product['name'] . ' (ID: ' . $product['id_product'] . ')';
                }
                unset($product);
                $productFieldSwitch = [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => sprintf('%1$s %2$s', $this->module->l('Filter by'), $this->module->l('Product')),
                    'name' => 'switch_products',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'switch_products_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'switch_products_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Display the popup only if the screen width is smaller than the specified number of pixels.'),
                ];
                $productField = [
                    'type' => 'swap-custom',
                    'label' => $this->module->l('Show Only to Selected Products'),
                    'name' => 'products[]',
                    'class' => 'switch_products',
                    'search' => true,
                    'options' => [
                        'query' => $products,
                        'id' => 'id_product',
                        'name' => 'display_name',
                    ],
                    'desc' => $this->module->l('Select the products where the popup should be displayed.'),
                ];

                array_push($this->fields_form[$fieldsFormIndex]['form']['input'], $productFieldSwitch, $productField);
            }
        }

        array_push(
            $this->fields_form[$fieldsFormIndex]['form']['input'],
            [
                'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                'label' => $this->module->l('Exclude Products with Specific Price'),
                'name' => 'product_specific_price',
                'class' => 't',
                'col' => 8,
                'is_bool' => true,
                'values' => [
                    [
                        'id' => 'product_specific_price_on',
                        'value' => 1,
                        'label' => $this->module->l('Enabled'),
                    ],
                    [
                        'id' => 'product_specific_price_off',
                        'value' => 0,
                        'label' => $this->module->l('Disabled'),
                    ],
                ],
                'desc' => $this->module->l('Prevent the popup from appearing in products that have a specific price.'),
            ],
            [
                'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                'label' => sprintf('%1$s %2$s', $this->module->l('Filter by'), $this->module->l('Product Stock')),
                'name' => 'product_stock',
                'class' => 't',
                'col' => 8,
                'is_bool' => true,
                'values' => [
                    [
                        'id' => 'product_stock_on',
                        'value' => 1,
                        'label' => $this->module->l('Enabled'),
                    ],
                    [
                        'id' => 'product_stock_off',
                        'value' => 0,
                        'label' => $this->module->l('Disabled'),
                    ],
                ],
                'desc' => $this->module->l('Display the popup only for products with specific stock availability.'),
            ],
            [
                'type' => 'text',
                'label' => $this->module->l('From'),
                'name' => 'product_stock_from',
                'prefix' => '>=',
                'col' => 8,
                'desc' => $this->module->l('Set the minimum stock availability for the popup to be displayed.'),
            ],
            [
                'type' => 'text',
                'label' => $this->module->l('To'),
                'name' => 'product_stock_to',
                'prefix' => '<=',
                'col' => 8,
                'desc' => $this->module->l('Set the maximum stock availability for the popup to be displayed.'),
            ],
            [
                'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                'label' => sprintf('%1$s %2$s', $this->module->l('Filter by'), $this->module->l('Product price')),
                'name' => 'product_price',
                'class' => 't',
                'col' => 8,
                'is_bool' => true,
                'values' => [
                    [
                        'id' => 'product_price_on',
                        'value' => 1,
                        'label' => $this->module->l('Enabled'),
                    ],
                    [
                        'id' => 'product_price_off',
                        'value' => 0,
                        'label' => $this->module->l('Disabled'),
                    ],
                ],
                'desc' => $this->module->l('Display the popup only for products within a specific price range.'),
            ],
            [
                'type' => 'text',
                'label' => $this->module->l('From'),
                'name' => 'product_price_from',
                'prefix' => '>=',
                'suffix' => version_compare(_PS_VERSION_, '1.7.6', '>=') ? $this->context->currency->symbol : $this->context->currency->sign,
                'col' => 8,
                'desc' => $this->module->l('Set the minimum price for the popup to be displayed.'),
            ],
            [
                'type' => 'text',
                'label' => $this->module->l('To'),
                'name' => 'product_price_to',
                'prefix' => '<=',
                'suffix' => version_compare(_PS_VERSION_, '1.7.6', '>=') ? $this->context->currency->symbol : $this->context->currency->sign,
                'col' => 8,
                'desc' => $this->module->l('Set the maximum price for the popup to be displayed.'),
            ],
            [
                'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                'label' => sprintf('%1$s "%2$s"', $this->module->l('Filter by'), $this->module->l('When out of stock')),
                'name' => 'product_oos',
                'class' => 't',
                'col' => 8,
                'is_bool' => true,
                'values' => [
                    [
                        'id' => 'product_price_on',
                        'value' => 1,
                        'label' => $this->module->l('Enabled'),
                    ],
                    [
                        'id' => 'product_price_off',
                        'value' => 0,
                        'label' => $this->module->l('Disabled'),
                    ],
                ],
                'desc' => $this->module->l('Display the popup only for products within a specific price range.'),
            ],
            [
                'type' => 'radio',
                'label' => sprintf('"%1$s" %2$s', $this->module->l('When out of stock'), $this->module->l('value')),
                'name' => 'product_oos_value',
                'class' => 't',
                'col' => 8,
                'is_bool' => true,
                'values' => [
                    [
                        'id' => 'product_oos_value_1',
                        'value' => 0,
                        'label' => $this->module->l('Deny orders'),
                    ],
                    [
                        'id' => 'product_oos_value_2',
                        'value' => 1,
                        'label' => $this->module->l('Allow orders'),
                    ],
                ],
                'desc' => $this->module->l('Display the popup only for products within a specific price range.'),
            ]
        );

        if (!Configuration::get('APC_HIDE_FILTER_ATTRIBUTE')) {
            if (Combination::isFeatureActive()) {
                $attributes = self::getAttributes((int) $this->context->language->id);
                if ($attributes) {
                    foreach ($attributes as &$attribute) {
                        $attribute['display_name'] = $attribute['public_name'] . ' - ' . $attribute['name'] . ' (ID: ' . $attribute['id_attribute'] . ')';
                    }
                    unset($attribute);

                    if (count($attributes)) {
                        array_push(
                            $this->fields_form[$fieldsFormIndex]['form']['input'],
                            [
                                'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                                'label' => sprintf('%1$s %2$s', $this->module->l('Filter by'), $this->module->l('Attribute')),
                                'name' => 'switch_attributes',
                                'class' => 't',
                                'col' => 8,
                                'is_bool' => true,
                                'values' => [
                                    [
                                        'id' => 'switch_attribute_on',
                                        'value' => 1,
                                        'label' => $this->module->l('Enabled'),
                                    ],
                                    [
                                        'id' => 'switch_attribute_off',
                                        'value' => 0,
                                        'label' => $this->module->l('Disabled'),
                                    ],
                                ],
                                'desc' => $this->module->l('Display the popup only for products with the selected attributes.'),
                            ],
                            [
                                'type' => 'swap-custom',
                                'label' => $this->module->l('Show Only to Selected Attributes'),
                                'name' => 'attributes[]',
                                'class' => 'switch_attributes if_value_switch_attributes',
                                'search' => true,
                                'sort' => 'display_name',
                                'options' => [
                                    'query' => $attributes,
                                    'id' => 'id_attribute',
                                    'name' => 'display_name',
                                ],
                                'desc' => $this->module->l('Select the attributes to filter products for displaying the popup.'),
                            ]
                        );
                    }
                }
            }
        }

        if (!Configuration::get('APC_HIDE_FILTER_FEATURE')) {
            if (Feature::isFeatureActive()) {
                $features = [];
                $featureGroups = Feature::getFeatures((int) $this->context->language->id);
                foreach ($featureGroups as $featureGroup) {
                    $featuresValue = FeatureValue::getFeatureValuesWithLang((int) $this->context->language->id, $featureGroup['id_feature']);
                    foreach ($featuresValue as $featureValue) {
                        $featureValue['display_name'] = $featureGroup['name'] . ' - ' . $featureValue['value'] . ' (ID: ' . $featureValue['id_feature_value'] . ')';
                        array_push($features, $featureValue);
                    }
                }

                if (count($features)) {
                    array_push(
                        $this->fields_form[$fieldsFormIndex]['form']['input'],
                        [
                            'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                            'label' => sprintf('%1$s %2$s', $this->module->l('Filter by'), $this->module->l('Feature')),
                            'name' => 'switch_features',
                            'class' => 't',
                            'col' => 8,
                            'is_bool' => true,
                            'values' => [
                                [
                                    'id' => 'switch_features_on',
                                    'value' => 1,
                                    'label' => $this->module->l('Enabled'),
                                ],
                                [
                                    'id' => 'switch_features_off',
                                    'value' => 0,
                                    'label' => $this->module->l('Disabled'),
                                ],
                            ],
                            'desc' => $this->module->l('Display the popup only for products with the selected features.'),
                        ],
                        [
                            'type' => 'swap-custom',
                            'label' => $this->module->l('Show Only to Selected Features'),
                            'name' => 'features[]',
                            'class' => 'switch_features',
                            'search' => true,
                            'sort' => 'display_name',
                            'options' => [
                                'query' => $features,
                                'id' => 'id_feature_value',
                                'name' => 'display_name',
                            ],
                            'desc' => $this->module->l('Select the features to filter products for displaying the popup.'),
                        ]
                    );
                }
            }
        }

        if (!Configuration::get('APC_HIDE_FILTER_CATEGORY')) {
            array_push(
                $this->fields_form[$fieldsFormIndex]['form']['input'],
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => sprintf('%1$s %2$s', $this->module->l('Filter by'), $this->module->l('Category')),
                    'name' => 'switch_categories',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'switch_categories_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'switch_categories_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Display the popup only in the selected categories or for products within these categories.'),
                ],
                [
                    'type' => 'swap-custom',
                    'label' => $this->module->l('Show Only to Selected Categories or to the products that belong to the selected categories'),
                    'name' => 'categories[]',
                    'class' => 'switch_categories',
                    'search' => true,
                    'options' => [
                        'query' => $categories,
                        'id' => 'id_category',
                        'name' => 'display_name',
                    ],
                    'desc' => $this->module->l('Select categories or products within them to display the popup.'),
                ]
            );
        }

        if (!Configuration::get('APC_HIDE_FILTER_MANUFACTURER')) {
            $manufacturers = Manufacturer::getManufacturers(false, $id_lang, false);
            if ($manufacturers) {
                $manufacturerFieldSwitch = [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => sprintf('%1$s %2$s', $this->module->l('Filter by'), $this->module->l('Manufacturer')),
                    'name' => 'switch_manufacturers',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'switch_manufacturers_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'switch_manufacturers_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Display the popup only in the selected manufacturers or for products associated with these manufacturers.'),
                ];

                $manufacturerField = [
                    'type' => 'swap-custom',
                    'label' => $this->module->l('Show Only to Selected Manufacturers or to the products that belong to the selected manufacturers'),
                    'name' => 'manufacturers[]',
                    'class' => 'switch_manufacturers',
                    'search' => true,
                    'options' => [
                        'query' => $manufacturers,
                        'id' => 'id_manufacturer',
                        'name' => 'name',
                    ],
                    'desc' => $this->module->l('Select manufacturers or products associated with them to display the popup.'),
                ];

                array_push($this->fields_form[$fieldsFormIndex]['form']['input'], $manufacturerFieldSwitch, $manufacturerField);
            }
        }

        if (!Configuration::get('APC_HIDE_FILTER_SUPPLIER')) {
            $suppliers = Supplier::getSuppliers(false, $id_lang, false);
            if (count($suppliers)) {
                $supplierFieldSwitch = [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => sprintf('%1$s %2$s', $this->module->l('Filter by'), $this->module->l('Supplier')),
                    'name' => 'switch_suppliers',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'switch_suppliers_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'switch_suppliers_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Display the popup only in the selected suppliers or for products supplied by them.'),
                ];

                $supplierField = [
                    'type' => 'swap-custom',
                    'label' => $this->module->l('Show Only to Selected Suppliers or to the products that belong to the selected suppliers'),
                    'name' => 'suppliers[]',
                    'class' => 'switch_suppliers',
                    'search' => true,
                    'options' => [
                        'query' => $suppliers,
                        'id' => 'id_supplier',
                        'name' => 'name',
                    ],
                    'desc' => $this->module->l('Select suppliers or products associated with them to display the popup.'),
                ];

                array_push($this->fields_form[$fieldsFormIndex]['form']['input'], $supplierFieldSwitch, $supplierField);
            }
        }

        array_push(
            $this->fields_form[$fieldsFormIndex]['form']['input'],
            [
                'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                'label' => sprintf('%1$s %2$s', $this->module->l('Filter by'), $this->module->l('Page/Controller')),
                'name' => 'switch_controller_exceptions',
                'class' => 't',
                'col' => 8,
                'is_bool' => true,
                'values' => [
                    [
                        'id' => 'switch_controller_exceptions_on',
                        'value' => 1,
                        'label' => $this->module->l('Enabled'),
                    ],
                    [
                        'id' => 'switch_controller_exceptions_off',
                        'value' => 0,
                        'label' => $this->module->l('Disabled'),
                    ],
                ],
                'desc' => $this->module->l('Display the popup only on the selected pages or controllers.'),
            ],
            [
                'type' => 'swap-custom',
                'label' => $this->module->l('Show Only to Selected Pages/Controllers'),
                'name' => 'controller_exceptions[]',
                'class' => 'switch_controller_exceptions',
                'search' => true,
                'options' => [
                    'query' => $list_controllers,
                    'id' => 'id',
                    'name' => 'name',
                ],
                'desc' => $this->module->l('Select specific pages or controllers where the popup should appear.'),
            ],
            [
                'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                'label' => sprintf('%1$s %2$s', $this->module->l('Filter by'), $this->module->l('CMS')),
                'name' => 'switch_cms',
                'class' => 't',
                'col' => 8,
                'is_bool' => true,
                'values' => [
                    [
                        'id' => 'switch_cms_on',
                        'value' => 1,
                        'label' => $this->module->l('Enabled'),
                    ],
                    [
                        'id' => 'switch_cms_off',
                        'value' => 0,
                        'label' => $this->module->l('Disabled'),
                    ],
                ],
                'desc' => $this->module->l('Display the popup only on the selected CMS pages.'),
            ],
            [
                'type' => 'swap-custom',
                'label' => $this->module->l('Show Only to Selected CMS'),
                'name' => 'cms[]',
                'class' => 'switch_cms',
                'search' => true,
                'options' => [
                    'query' => CMS::getCMSPages($this->context->language->id, null, false, $this->context->shop->id),
                    'id' => 'id_cms',
                    'name' => 'meta_title',
                ],
                'desc' => $this->module->l('Select the CMS pages where the popup should appear.'),
            ]
        );

        $this->fields_form[$fieldsFormIndex]['form']['submit'] = [
            'title' => $this->module->l('Save'),
            'type' => 'submit',
        ];

        ++$fieldsFormIndex;
        $this->fields_form[$fieldsFormIndex]['form'] = [
            'legend' => [
                'title' => $this->module->l('Sticky block'),
                'icon' => 'icon-anchor',
            ],
            'input' => [
                [
                    'type' => (version_compare(_PS_VERSION_, '1.6', '>=')) ? 'switch' : 'radio',
                    'label' => $this->module->l('Display a Sticky Block After Closing the Popup'),
                    'name' => 'sticky_display_close',
                    'class' => 't',
                    'col' => 8,
                    'is_bool' => true,
                    'values' => [
                        [
                            'id' => 'sticky_display_close_on',
                            'value' => 1,
                            'label' => $this->module->l('Enabled'),
                        ],
                        [
                            'id' => 'sticky_display_close_off',
                            'value' => 0,
                            'label' => $this->module->l('Disabled'),
                        ],
                    ],
                    'desc' => $this->module->l('Show a fixed-position block on the screen when the popup is closed'),
                ],
                [
                    'type' => 'select',
                    'label' => $this->module->l('Position'),
                    'name' => 'sticky_position',
                    'class' => 't',
                    'col' => 8,
                    'options' => [
                        'query' => [
                            [
                                'id' => '1',
                                'name' => $this->module->l('Top left'),
                            ],
                            [
                                'id' => '2',
                                'name' => $this->module->l('Top center'),
                            ],
                            [
                                'id' => '3',
                                'name' => $this->module->l('Top right'),
                            ],
                            [
                                'id' => '4',
                                'name' => $this->module->l('Center left'),
                            ],
                            [
                                'id' => '6',
                                'name' => $this->module->l('Center right'),
                            ],
                            [
                                'id' => '7',
                                'name' => $this->module->l('Bottom left'),
                            ],
                            [
                                'id' => '8',
                                'name' => $this->module->l('Bottom center'),
                            ],
                            [
                                'id' => '9',
                                'name' => $this->module->l('Bottom right'),
                            ],
                        ],
                        'id' => 'id',
                        'name' => 'name',
                    ],
                    'desc' => $this->module->l('Choose where on the screen the block will appear.'),
                ],
                [
                    'type' => 'textarea',
                    'label' => $this->module->l('Content for Sticky Block'),
                    'name' => 'sticky_content',
                    'id' => 'sticky_content',
                    'lang' => true,
                    'autoload_rte' => version_compare(_PS_VERSION_, '1.6', '>=') ? '' : true,
                    'cols' => 100,
                    'rows' => 10,
                    'class' => version_compare(_PS_VERSION_, '1.6', '>=') ? 'apc_tiny' : '',
                    'desc' => $this->module->l('Define the content that will appear in the sticky block.'),
                ],
                [
                    'type' => 'select',
                    'label' => $this->module->l('Content Rotation Angle'),
                    'name' => 'sticky_rotation',
                    'class' => 't',
                    'col' => 8,
                    'options' => [
                        'query' => [
                            [
                                'id' => '1',
                                'name' => '0%',
                            ],
                            [
                                'id' => '2',
                                'name' => '90%',
                            ],
                            [
                                'id' => '3',
                                'name' => '270%',
                            ],
                        ],
                        'id' => 'id',
                        'name' => 'name',
                    ],
                    'desc' => $this->module->l('Select the angle at which you want to rotate the content'),
                ],
            ],
            'buttons' => [
                'save-and-stay' => [
                    'title' => $this->module->l('Save and stay'),
                    'name' => 'submitAdd' . $this->table . 'AndStay',
                    'type' => 'submit',
                    'class' => 'btn btn-default pull-right',
                    'icon' => 'process-icon-save',
                ],
            ],
        ];

        ++$fieldsFormIndex;
        $this->context->smarty->assign([
            'previewLink' => Context::getContext()->link->getPageLink('index', true) . '?previewPopup=1&popupId=' . $object->id . '&token=' . Tools::getAdminToken('AdminAdvancedPopupCreatorPopups' . $object->id),
        ]);

        if (Validate::isLoadedObject($object)) {
            $this->fields_form[$fieldsFormIndex]['form'] = [
                'legend' => [
                    'title' => $this->module->l('Preview'),
                    'icon' => 'icon-photo',
                ],
                'input' => [
                    [
                        'type' => 'html',
                        'label' => ' ',
                        'col' => 12,
                        'name' => $this->context->smarty->fetch(_PS_MODULE_DIR_ . 'advancedpopupcreator/views/templates/admin/preview-notice.tpl'),
                    ],
                    [
                        'type' => 'html',
                        'label' => ' ',
                        'col' => 12,
                        'name' => $this->context->smarty->fetch(_PS_MODULE_DIR_ . 'advancedpopupcreator/views/templates/admin/preview-button.tpl'),
                    ],
                ],
            ];
        }

        // Multiselects
        $multiselects = [
            'controller_exceptions',
            'groups',
            'zones',
            'countries',
            'categories',
            'categories_selected',
            'manufacturers',
            'products',
            'genders',
            'customers',
            'suppliers',
            'cms',
            'languages',
            'attributes',
            'features',
        ];
        foreach ($multiselects as $multiselect) {
            $this->fields_value[$multiselect . '[]'] = explode(',', Tools::isSubmit('submitAdd' . $this->table) ? Tools::getValue($multiselect) : $object->$multiselect);
        }

        // Format dates
        if ($object->id) {
            if (strtotime($object->date_init) == 0) {
                $object->date_init = '';
            }

            if (strtotime($object->date_end) == 0) {
                $object->date_end = '';
            }
        }

        return parent::renderForm();
    }

    public function processSave()
    {
        // Multiselects
        $multiselects = [
            'controller_exceptions',
            'groups',
            'zones',
            'countries',
            'categories',
            'categories_selected',
            'manufacturers',
            'products',
            'genders',
            'customers',
            'suppliers',
            'cms',
            'languages',
            'attributes',
            'features',
        ];

        foreach ($multiselects as $multiselect) {
            $_POST[$multiselect] = Tools::getValue($multiselect) ? implode(',', Tools::getValue($multiselect)) : '';
        }

        $_POST['date_init'] = ((Tools::getValue('date_init') == '') ? date('Y-m-d H:i:s', 0) : Tools::getValue('date_init'));
        $_POST['date_end'] = ((Tools::getValue('date_end') == '') ? date('Y-m-d H:i:s', 0) : Tools::getValue('date_end'));

        parent::processSave();
    }

    protected function afterAdd($object)
    {
        return $this->afterUpdate($object);
    }

    protected function afterUpdate($object)
    {
        if (Validate::isLoadedObject($object)) {
            // Save file for each language
            foreach (Language::getLanguages(false) as $language) {
                if ($result = $this->module->uploadImage(_PS_MODULE_DIR_ . AdvancedPopupCreator::$image_dir, 'image_' . $language['id_lang'], $language['id_lang'], $object)) {
                    $object->image[$language['id_lang']] = $result;
                }

                if ($result = $this->module->uploadImage(_PS_MODULE_DIR_ . AdvancedPopupCreator::$image_dir, 'image_background_' . $language['id_lang'], $language['id_lang'], $object)) {
                    $object->image_background[$language['id_lang']] = $result;
                }
            }

            // Save
            $object->save();
        }

        return parent::afterUpdate($object);
    }

    protected function getProductsLite($id_lang, $only_active = false, $front = false)
    {
        $query = 'SELECT p.`id_product`, pl.`name`, product_shop.`id_shop`
                FROM `' . _DB_PREFIX_ . 'product` p
                ' . Shop::addSqlAssociation('product', 'p') . '
                LEFT JOIN `' . _DB_PREFIX_ . 'product_lang` pl ON (p.`id_product` = pl.`id_product` ' . Shop::addSqlRestrictionOnLang('pl') . ')
                WHERE pl.`id_lang` = ' . (int) $id_lang .
                    ($front ? ' AND product_shop.`visibility` IN ("both", "catalog")' : '') .
                    ($only_active ? ' AND product_shop.`active` = 1' : '');

        $rq = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);

        return $rq;
    }

    public static function getAttributes($idLang, $notNull = false)
    {
        if (!Combination::isFeatureActive()) {
            return [];
        }

        return Db::getInstance()->executeS('
            SELECT DISTINCT ag.*, agl.*, a.`id_attribute`, al.`name`, agl.`name` AS `attribute_group`
            FROM `' . _DB_PREFIX_ . 'attribute_group` ag
            LEFT JOIN `' . _DB_PREFIX_ . 'attribute_group_lang` agl
                ON (ag.`id_attribute_group` = agl.`id_attribute_group` AND agl.`id_lang` = ' . (int) $idLang . ')
            LEFT JOIN `' . _DB_PREFIX_ . 'attribute` a
                ON a.`id_attribute_group` = ag.`id_attribute_group`
            LEFT JOIN `' . _DB_PREFIX_ . 'attribute_lang` al
                ON (a.`id_attribute` = al.`id_attribute` AND al.`id_lang` = ' . (int) $idLang . ')
            ' . Shop::addSqlAssociation('attribute_group', 'ag') . '
            ' . Shop::addSqlAssociation('attribute', 'a') . '
            ' . ($notNull ? 'WHERE a.`id_attribute` IS NOT NULL AND al.`name` IS NOT NULL AND agl.`id_attribute_group` IS NOT NULL' : '') . '
            ORDER BY agl.`name` ASC, a.`position` ASC
        ');
    }

    protected function getNbProducts()
    {
        $query = 'SELECT count(*)
                FROM `' . _DB_PREFIX_ . 'product` p
                ' . Shop::addSqlAssociation('product', 'p');

        return (int) Db::getInstance(_PS_USE_SQL_SLAVE_)->getValue($query);
    }

    protected function getNbCustomers()
    {
        $query = 'SELECT count(*)
                FROM `' . _DB_PREFIX_ . 'customer`
                WHERE 1 ' . Shop::addSqlRestriction(Shop::SHARE_CUSTOMER);

        return (int) Db::getInstance(_PS_USE_SQL_SLAVE_)->getValue($query);
    }

    private function _createTemplate($tpl_name)
    {
        if ($this->override_folder) {
            if ($this->context->controller instanceof ModuleAdminController) {
                $override_tpl_path = $this->context->controller->getTemplatePath() . $tpl_name;
            } elseif ($this->module) {
                $override_tpl_path = _PS_MODULE_DIR_ . $this->module_name . '/views/templates/admin/' . $tpl_name;
            } else {
                if (file_exists($this->context->smarty->getTemplateDir(1) . DIRECTORY_SEPARATOR . $this->override_folder . $this->base_folder . $tpl_name)) {
                    $override_tpl_path = $this->context->smarty->getTemplateDir(1) . DIRECTORY_SEPARATOR . $this->override_folder . $this->base_folder . $tpl_name;
                } elseif (file_exists($this->context->smarty->getTemplateDir(0) . DIRECTORY_SEPARATOR . 'controllers' . DIRECTORY_SEPARATOR . $this->override_folder . $this->base_folder . $tpl_name)) {
                    $override_tpl_path = $this->context->smarty->getTemplateDir(0) . 'controllers' . DIRECTORY_SEPARATOR . $this->override_folder . $this->base_folder . $tpl_name;
                }
            }
        } elseif ($this->module) {
            $override_tpl_path = _PS_MODULE_DIR_ . $this->module_name . '/views/templates/admin/' . $tpl_name;
        }
        if (file_exists($override_tpl_path)) {
            return $this->context->smarty->createTemplate($override_tpl_path, $this->context->smarty);
        } else {
            return $this->context->smarty->createTemplate($tpl_name, $this->context->smarty);
        }
    }

    protected function getControllersExceptions($array)
    {
        $list_controllers = [];
        $i = 0;

        foreach ($array as $key => $value) {
            $meta = Meta::getMetaByPage($key, (int) $this->context->cookie->id_lang);

            $list_controllers[$i]['id'] = $key;
            $list_controllers[$i]['value'] = $key;
            if ($meta && $meta['title']) {
                $list_controllers[$i]['name'] = $key . ' - ' . $meta['title'] . ' (' . $meta['url_rewrite'] . ')';
            } elseif ($key == 'auth') {
                $list_controllers[$i]['name'] = $key . ' - ' . $this->module->l('Authentication');
            } elseif ($key == 'category') {
                $list_controllers[$i]['name'] = $key . ' - ' . $this->module->l('Category page');
            } elseif ($key == 'category') {
                $list_controllers[$i]['name'] = $key . ' - ' . $this->module->l('CMS page');
            } elseif ($key == 'index') {
                $list_controllers[$i]['name'] = $key . ' - ' . $this->module->l('Home page');
            } elseif ($key == 'myaccount') {
                $list_controllers[$i]['name'] = $key . ' - ' . $this->module->l('My account page');
            } elseif ($key == 'orderopc') {
                $list_controllers[$i]['name'] = $key . ' - ' . $this->module->l('Checkout');
            } elseif ($key == 'product') {
                $list_controllers[$i]['name'] = $key . ' - ' . $this->module->l('Product page');
            } else {
                $list_controllers[$i]['name'] = $value;
            }

            ++$i;
        }

        return $list_controllers;
    }

    public static function getModuleControllers($type = 'all')
    {
        $modules_controllers = [];
        $modules = Module::getModulesDirOnDisk();

        foreach ($modules as $module) {
            foreach (Dispatcher::getControllersInDirectory(_PS_MODULE_DIR_ . $module . '/controllers/') as $controller) {
                if ($type == 'front') {
                    if (strpos($controller, 'Admin') === false) {
                        $modules_controllers['module' . $module . $controller] = $controller . ' - Module: ' . $module;
                    }
                } else {
                    $modules_controllers['module' . $module . $controller] = $controller . ' - Module: ' . $module;
                }
            }
        }

        return $modules_controllers;
    }

    public function printValidDate($value)
    {
        if (strtotime($value)) {
            return $value;
        }

        return '∞';
    }

    public function printValidIcon($value, $object)
    {
        $today = date('Y-m-d H:i:s');

        if ($object['date_init'] > $today) {
            $dateTitle = $this->module->l('Future rule');
            if (strtotime($object['date_init']) > 0) {
                $dateTitle .= '. ' . $this->module->l('Begins in:') . ' ' . $object['date_init'];
            }

            $iconClass = 'future-date-icon';
        }

        if (strtotime($object['date_end']) == 0 || $today < $object['date_end']) {
            $dateTitle = $this->module->l('Valid rule');
            if (strtotime($object['date_init']) > 0 && strtotime($object['date_end']) > 0) {
                $dateTitle .= '. ' . $this->module->l('From:') . ' ' . $object['date_init'] . '. ' . $this->module->l('Until:') . ' ' . $object['date_end'];
            } elseif (strtotime($object['date_init']) > 0 && strtotime($object['date_end']) == 0) {
                $dateTitle .= '. ' . $this->module->l('From:') . ' ' . $object['date_init'] . ' (' . $this->module->l('no expires') . ')';
            } elseif (strtotime($object['date_init']) == 0 && strtotime($object['date_end']) > 0) {
                $dateTitle .= '. ' . $this->module->l('Until:') . ' ' . $object['date_end'];
            } else {
                $dateTitle .= ' (' . $this->module->l('no expires') . ')';
            }

            $iconClass = 'valid-date-icon';
        } else {
            $dateTitle = $this->module->l('Expired rule');
            if (strtotime($object['date_init']) > 0 && strtotime($object['date_end']) > 0) {
                $dateTitle .= '. ' . $this->module->l('Between:') . ' ' . $object['date_init'] . ' ' . $this->module->l('and:') . ' ' . $object['date_end'];
            } else {
                $dateTitle .= '. ' . $this->module->l('From:') . ' ' . $object['date_end'];
            }

            $iconClass = 'expired-date-icon';
        }

        $this->context->smarty->assign([
            'dateTitle' => $dateTitle,
            'iconClass' => $iconClass,
        ]);

        return $this->context->smarty->fetch($this->module->getLocalPath() . 'views/templates/admin/valid.tpl');
    }

    public function printCheckIcon($value, $object)
    {
        return $value ? '<span class="list-action-enable action-enabled"><i class="icon-check"></i></span>' : '<span class="list-action-enable action-disabled"><i class="icon-remove"></i></span>';
    }

    public function displayDuplicateLink($token = null, $id = null)
    {
        if (version_compare(_PS_VERSION_, '1.6', '<')) {
            $tpl = $this->createTemplate('list_action_duplicate_15.tpl');

            $tpl->assign([
                'href' => self::$currentIndex . '&' . $this->identifier . '=' . $id . '&duplicate' . $this->table . '&token=' . ($token != null ? $token : $this->token),
                'action' => $this->module->l('Duplicate'),
            ]);
        } else {
            $tpl = $this->createTemplate('list_action_duplicate.tpl');
            if (!array_key_exists('Bad SQL query', self::$cache_lang)) {
                self::$cache_lang['Duplicate'] = $this->module->l('Duplicate', 'Helper');
            }

            $duplicate = self::$currentIndex . '&' . $this->identifier . '=' . $id . '&duplicate' . $this->table;

            $tpl->assign([
                'href' => self::$currentIndex . '&' . $this->identifier . '=' . $id . '&duplicate' . $this->table . '&token=' . ($token != null ? $token : $this->token),
                'action' => self::$cache_lang['Duplicate'],
                'location' => $duplicate . '&token=' . ($token != null ? $token : $this->token),
            ]);
        }

        return $tpl->fetch();
    }

    public function processDuplicate()
    {
        $advancedPopup = new AdvancedPopup((int) Tools::getValue('id_advancedpopup'));
        if (Validate::isLoadedObject($advancedPopup)) {
            unset($advancedPopup->id);
            $advancedPopup->active = 0;

            if ($advancedPopup->add()) {
                // Copy files
                foreach (Language::getLanguages(false) as $language) {
                    if ($advancedPopup->image[$language['id_lang']]) {
                        $oldImg = _PS_MODULE_DIR_ . AdvancedPopupCreator::$image_dir . $advancedPopup->image[$language['id_lang']];
                        $pathInfo = pathinfo($oldImg);
                        $newImgFileName = Tools::substr(str_shuffle(md5(microtime())), 0, 15) . '.' . $pathInfo['extension'];
                        $newImg = _PS_MODULE_DIR_ . AdvancedPopupCreator::$image_dir . $newImgFileName;

                        if (!copy($oldImg, $newImg)) {
                            echo 'failed to copy';
                        }

                        $advancedPopup->image[$language['id_lang']] = $newImgFileName;
                    }

                    if ($advancedPopup->image_background[$language['id_lang']]) {
                        $oldImg = _PS_MODULE_DIR_ . AdvancedPopupCreator::$image_dir . $advancedPopup->image_background[$language['id_lang']];
                        $pathInfo = pathinfo($oldImg);
                        $newImgFileName = Tools::substr(str_shuffle(md5(microtime())), 0, 15) . '.' . $pathInfo['extension'];
                        $newImg = _PS_MODULE_DIR_ . AdvancedPopupCreator::$image_dir . $newImgFileName;

                        if (!copy($oldImg, $newImg)) {
                            echo 'failed to copy';
                        }

                        $advancedPopup->image_background[$language['id_lang']] = $newImgFileName;
                    }
                }

                $advancedPopup->save();

                $this->redirect_after = self::$currentIndex . (Tools::getIsset('id_advancedpopup') ? '&id_advancedpopup=' . (int) Tools::getValue('id_advancedpopup') : '') . '&conf=19&token=' . $this->token;
            } else {
                $this->errors[] = Tools::displayError('An error occurred while creating an object.');
            }
        }
    }
}
